package csdk.v2.runner.filesystem;

import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import csdk.v2.runner.Runner;

/**
 * Lista dos tipos de arquivo conhecidos pelo {@link Runner}
 *
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class FileTypes {

  /**
   * Tipo do arquivo que indica "Arquivo Desconhecido"
   */
  public static final String UNKNOWN = "UNKNOWN";

  /**
   * Tipo do arquivo que indica diretrio.
   */
  public static final String DIRECTORY_TYPE = "DIRECTORY_TYPE";

  /**
   * Lista de tipos de arquivos.
   */
  static private Map<String, FileType> types = new HashMap<String, FileType>();

  /**
   * Cria um tipo de arquivo para o {@link Runner}.
   *
   * @param typeName o nome do novo tipo.
   * @return o tipo criado.
   */
  public static FileType createFileType(String typeName) {
    if (hasFileType(typeName)) {
      String pattern = "Type {0} already exists!";
      String err = MessageFormat.format(pattern, typeName);
      throw new IllegalArgumentException(err);
    }
    FileType typ = new FileType(typeName);
    types.put(getFileTypeKey(typeName), typ);
    return typ;
  }

  /**
   * Indica se um tipo j existe.
   *
   * @param fileTypeName o tipo
   * @return indicativo
   */
  static public FileType getFileType(String fileTypeName) {
    if (fileTypeName == null) {
      return null;
    }
    String typeName = fileTypeName.trim();
    if (typeName.equals(UNKNOWN)) {
      return null;
    }

    return types.get(getFileTypeKey(fileTypeName));
  }

  /**
   * Indica se um tipo j existe.
   *
   * @param typeName o tipo
   * @return indicativo
   */
  static public boolean hasFileType(String typeName) {
    return getFileType(typeName) != null;
  }

  /**
   * Busca um tipo de arquivo aplicvel a extenso sugerida.
   *
   * @param extension extenso
   * @return o tipo
   */
  static public FileType getFileTypeFromExtension(String extension) {
    if (extension == null) {
      throw new RuntimeException("Internal error: null extension not allowed");
    }
    for (FileType value : types.values()) {
      ArrayList<String> extensions = value.getExtensions();
      for (String ext : extensions) {
        if (extension.trim().equalsIgnoreCase((ext.trim()))) {
          return value;
        }
      }
    }
    return null;
  }

  /**
   * Busca a extenso sugerida com base em um tipo de arquivo aplicvel.
   *
   * @param typeName o tipo
   * @return as extenses associadas ao tipo
   */
  static public String[] getExtensionsFromFileType(String typeName) {
    if (typeName == null) {
      throw new RuntimeException("Internal error: null type name not allowed");
    }
    if (typeName.equals(UNKNOWN)) {
      return null;
    }

    FileType fileType = types.get(getFileTypeKey(typeName));
    if (fileType == null) {
      return null;
    }

    ArrayList<String> extensions = fileType.getExtensions();
    return extensions.toArray(new String[extensions.size()]);
  }

  /**
   * Obtm a chave para mapeamento do tipo.
   *
   * @param typeName o nome do tipo.
   * @return a chave.
   */
  private static String getFileTypeKey(String typeName) {
    return typeName.trim().toLowerCase();
  }

  /**
   * Carrega as propriedades que definem os tipos de arquivo.
   *
   * @param input o input stream das propriedades.
   */
  public static void loadFileTypes(InputStream input) {
    types.clear();
    Properties properties = loadProperties(input);
    for (int i = 0; true; i++) {
      FileType fileType = makeFileType(properties, i);
      if (fileType == null) {
        break;
      }
      types.put(getFileTypeKey(fileType.getName()), fileType);
    }
  }

  /**
   * Constri um tipo de arquivo a partir das propriedades especificadas.
   *
   * @param properties .
   * @param id .
   *
   * @return .
   */
  private static FileType makeFileType(Properties properties, int id) {
    final String typeCode = getTypeCode(id, properties);
    // Se o type code no for encontrado no existe mais tipos
    if (typeCode == null) {
      return null;
    }
    try {
      List<String> extensions = getExtensions(typeCode, properties);
      FileType fileType = new FileType(typeCode);
      if (extensions != null) {
        for (String extension : extensions) {
          fileType.addExtension(extension);
        }
      }
      return fileType;
    }
    catch (Exception e) {
      throw new RuntimeException("Error reading file type " + id);
    }
  }

  /**
   * Obtm o valor da propriedade que possui as extensoes de um tipo de arquivo.
   *
   * @param typeCode .
   * @param properties .
   *
   * @return .
   */
  private static List<String> getExtensions(String typeCode,
    Properties properties) {
    String propName = "filetype." + typeCode + ".extensions";
    String extensions = properties.getProperty(propName);
    if (extensions == null) {
      return null;
    }
    String[] extS = extensions.split(",");
    List<String> extVector = new ArrayList<String>(extS.length);
    for (int i = 0; i < extS.length; i++) {
      extVector.add(extS[i].trim().toLowerCase());
    }
    return extVector;
  }

  /**
   * Obtm o valor da propriedade que possui o identificador de um tipo de
   * arquivo.
   *
   * @param i .
   * @param properties .
   *
   * @return .
   */
  private static String getTypeCode(int i, Properties properties) {
    String propName = "filetype." + i + ".typeCode";
    final String prop = properties.getProperty(propName);
    if (prop == null) {
      return null;
    }
    final String code = prop.trim();
    return code;
  }

  /**
   * Carrega o arquivo de proprieades.
   *
   * @param in o input stream do arquivo.
   *
   * @return as propriedades.
   */
  private static Properties loadProperties(InputStream in) {
    if (in == null) {
      throw new IllegalArgumentException("Input stream cannot be null");
    }
    try {
      Properties properties = new Properties();
      properties.load(in);
      return properties;
    }
    catch (IOException e) {
      throw new RuntimeException("Error reading file types list resource", e);
    }
    finally {
      if (in != null) {
        try {
          in.close();
        }
        catch (IOException e) {
        }
      }
    }
  }
}
