package csdk.v2.runner.application;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.Map;

import csdk.v2.api.application.ApplicationException;
import csdk.v2.api.application.IApplication;
import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.application.IMessage;
import csdk.v2.api.application.IMessageSender;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.core.IContext;
import csdk.v2.api.filesystem.project.IProjectContext;
import csdk.v2.api.filesystem.project.IProjectObserver;
import csdk.v2.runner.ApplicationManager;
import csdk.v2.runner.ApplicationRegistry;
import csdk.v2.runner.CSDKLogger;
import csdk.v2.runner.IContextFactory;
import csdk.v2.runner.Runner;
import csdk.v2.runner.core.RunnerEnvironment;

/**
 * Classe que mapeia uma aplicao baseada no CSDK para a sandbox.
 *
 * Essa classe *no* deve ser usada por desenvolvedores CSDK em suas aplicaes.
 * Ela  de uso exclusivo do ambiente simulado do {@link Runner}.
 */
public class RunnerApplication implements IApplication {

  /**
   * Contador usado para gerar os identificadores das instncias.
   */
  private static int instanceCounter = 0;

  /**
   * A aplicao propriamente dita.
   */
  private final IApplication application;

  /**
   * A interface com o ambiente CSDK do {@link Runner}.
   */
  private final RunnerEnvironment runnerEnv;

  /**
   * Construtor.
   *
   * @param registry o registro da aplicao.
   * @param charset charset do ambiente.
   * @throws ApplicationException em caso de erro ao instanciar a aplicao.
   */
  public RunnerApplication(ApplicationRegistry registry, Charset charset)
    throws ApplicationException {
    String id = registry.getApplicationId();
    String instanceId = generateInstanceId(id);
    this.runnerEnv = createCSDKEnvironment(instanceId, registry, charset);
    Class<? extends IApplication> appClass = loadApplicationClass(registry);
    if (appClass == null) {
      String pattern = "No class found for {0}.";
      String message = MessageFormat.format(pattern, id);
      throw new ApplicationException(message);
    }
    try {
      Constructor<?> constructor =
        appClass.getConstructor(ICSDKEnvironment.class);
      this.application = (IApplication) constructor.newInstance(this.runnerEnv);
    }
    catch (final InvocationTargetException e) {
      final Throwable appException = e.getTargetException();
      String pattern =
        "Error instantiating application {0} main class {1}.\nMessage: {2}";
      String message =
        MessageFormat.format(pattern, id, appClass.getName(), appException
          .getLocalizedMessage());
      throw new ApplicationException(message, appException);
    }
    catch (final NoSuchMethodException e) {
      String pattern =
        "Application {0} is not configured correctly or has implementation problems.\nThe main class {1} has no public valid constructor.";
      String message = MessageFormat.format(pattern, id, appClass.getName());
      throw new ApplicationException(message, e);
    }
    catch (final InstantiationException e) {
      String pattern =
        "Application {0} has instanciation problems.\nPossible cause: Application main class {1} is abstract.";
      String message = MessageFormat.format(pattern, id, appClass.getName());
      throw new ApplicationException(message, e);
    }
    catch (final IllegalAccessException e) {
      String pattern =
        "Application {0} has implementation problems in its main class {1}.";
      String message = MessageFormat.format(pattern, id, appClass.getName());
      throw new ApplicationException(message, e);
    }
  }

  /**
   * Faz a carga da classe (com base no nome). Esta funo utiliza
   * <b>extraordinariamente</b> a anotao de supresso de avisos (warnings)
   * porque foi necessrio fazer um <i>loadClass</i>.
   *
   * (Ver definio de {@link ClassLoader#loadClass(String)})
   *
   * @param reg registro da aplicao.
   *
   * @return a classe.
   * @throws ApplicationException se a classe da aplicao no for encontrada ou
   *         vlida.
   */
  @SuppressWarnings("unchecked")
  private Class<? extends IApplication> loadApplicationClass(
    ApplicationRegistry reg) throws ApplicationException {
    try {
      ClassLoader classloader = reg.getClassloader();
      final Class<?> loadedClass =
        Class.forName(reg.getClassName(), true, classloader);
      if (IApplication.class.isAssignableFrom(loadedClass)) {
        return (Class<? extends IApplication>) loadedClass;
      }
      else {
        String pattern = "Aplication main class must implement interface {0}.";
        String message =
          MessageFormat.format(pattern, IApplication.class.getName());
        throw new IllegalArgumentException(message);
      }
    }
    catch (Exception e) {
      throw new ApplicationException("Error loading application main class", e);
    }
  }

  /**
   * Cria o ambiente de interface com o CSDK para a aplicao.
   *
   * @param instanceId identificador da instncia.
   * @param registry o registro da aplicao.
   * @param charset charset do ambiente.
   * @return o ambiente.
   *
   * @throws ApplicationException em caso de erro ao criar o ambiente.
   */
  private RunnerEnvironment createCSDKEnvironment(String instanceId,
    ApplicationRegistry registry, Charset charset) throws ApplicationException {
    final ApplicationManager appManager = ApplicationManager.getInstance();
    IContextFactory contextFactory = appManager.getContextFactory();
    Map<Class<? extends IContext>, IContext> contexts =
      contextFactory.createCSDKContexts(instanceId, registry, appManager
        .getRunnerProperties());
    if (contexts == null || contexts.isEmpty()
      || !contexts.containsKey(IApplicationContext.class)) {
      throw new ApplicationException(
        "Context factory must create the mandatory context "
          + IApplicationContext.class.getSimpleName());
    }
    if (registry.requiresProject()) {
      if (!contexts.containsKey(IProjectContext.class)) {
        throw new ApplicationException(
          "Application cannot be run without a project context ("
            + IProjectContext.class.getSimpleName() + ")");
      }
      else {
        IProjectContext projectContext =
          (IProjectContext) contexts.get(IProjectContext.class);
        projectContext.addProjectObserver(new IProjectObserver() {
          @Override
          public void onProjectOpen(String projectId) {
            //ignora evento
          }

          @Override
          public void onProjectClose(String projectId) {
            finishApplication();
          }
        });
      }
    }
    return new RunnerEnvironment(contexts, charset);
  }

  /**
   * Gera o identificador nico da instncia da aplicao.
   *
   * @param id o identificador da aplicao.
   * @return o identificador da instncia.
   */
  private synchronized static String generateInstanceId(String id) {
    return id + "_" + ++instanceCounter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    return application.canEndApplication();
  }

  /**
   * Tenta terminar a execuo da aplicao
   *
   * @return verdadeiro se a execuo foi terminada.
   */
  public boolean finishApplication() {
    if (!canEndApplication()) {
      return false;
    }
    killApplication();
    return true;
  }

  /**
   * Termina a execuo aplicao.
   */
  private void killApplication() {
    try {
      application.onApplicationEnd();
    }
    catch (ApplicationException e) {
      CSDKLogger logger = CSDKLogger.getInstance();
      String msg = "The finalization of the application generated an error.";
      logger.logSevere(msg);
      logger.logException(e);
    }
    finally {
      runnerEnv.cleanupContexts();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onAsyncMessageReceived(IMessage message, IMessageSender sender) {
    this.application.onAsyncMessageReceived(message, sender);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object onSyncMessageReceived(IMessage message, IMessageSender sender) {
    return this.application.onSyncMessageReceived(message, sender);
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationEnd() throws ApplicationException {
    this.killApplication();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onApplicationStart() throws ApplicationException {
    this.application.onApplicationStart();
  }

  /**
   * Obtm o id da instncia da aplicao.
   *
   * @return o id com uma string.
   */
  public String getInstanceId() {
    IApplicationContext context =
      runnerEnv.getContext(IApplicationContext.class);
    return context.getInstanceId();
  }

  /**
   * Obtm o id da aplicao.
   *
   * @return o id com uma string.
   */
  public String getApplicationId() {
    IApplicationContext context =
      runnerEnv.getContext(IApplicationContext.class);
    return context.getApplicationId();
  }

  /**
   * Obtm a aplicao propriamente dita.
   *
   * @return a aplicao.
   */
  public IApplication getApplication() {
    return application;
  }

}
