package csdk.v2.runner.core;

import java.awt.Window;
import java.io.Serializable;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;

import javax.swing.JDialog;

import csdk.v2.api.application.IApplicationContext;
import csdk.v2.api.core.ICSDKEnvironment;
import csdk.v2.api.core.IContext;
import csdk.v2.runner.ApplicationManager;
import csdk.v2.runner.Runner;
import csdk.v2.runner.application.RunnerApplication;

/**
 * Interface das aplicaes com o ambiente {@link Runner}.
 */
public class RunnerEnvironment implements ICSDKEnvironment, Serializable {

  /**
   * Contextos da aplicao.
   */
  private final Map<Class<? extends IContext>, IContext> contexts;

  /**
   * O charset do ambiente.
   */
  private final Charset charset;

  /**
   * Construtor.
   *
   * @param contexts contextos disponveis no ambiente.
   * @param charsetName charset do ambiente.
   */
  public RunnerEnvironment(Map<Class<? extends IContext>, IContext> contexts,
    Charset charsetName) {
    this.contexts = contexts;
    this.charset = charsetName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    IApplicationContext appContext = getApplicationContext();
    String instanceId = appContext.getInstanceId();
    final ApplicationManager appManager = ApplicationManager.getInstance();
    RunnerApplication app = appManager.getRunningApp(instanceId);
    if (app == null) {
      String pattern =
        "Application {0} cannot be finished, it is not active yet or has already been deactivated";
      String message = MessageFormat.format(pattern, instanceId);
      throw new IllegalStateException(message);
    }
    if (app.finishApplication()) {
      appManager.deactivate(app);
    }
  }

  /**
   * Obtm o contexto de aplicao do ambiente.
   *
   * @return o contexto de aplicao.
   */
  private IApplicationContext getApplicationContext() {
    return (IApplicationContext) this.contexts.get(IApplicationContext.class);
  }

  /**
   * Limpa os contextos.
   */
  public void cleanupContexts() {
    for (IContext context : this.contexts.values()) {
      if (context.isActive()) {
        context.onContextDeactivate();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleException(Exception error, Window parent) {
    IApplicationContext appContext = getApplicationContext();
    String[] additionalInfo = createEnvironmentInfo(appContext);
    JDialog dialog = new RunnerErrorDialog(parent, null, error, additionalInfo);
    dialog.setVisible(true);
  }

  /**
   * Cria o conjunto de informaes sobre o ambiente.
   *
   * @param appContext contexto de aplicao.
   *
   * @return a lista de strings de informaes.
   */
  private String[] createEnvironmentInfo(IApplicationContext appContext) {
    String applicationName = appContext.getApplicationName();
    String applicationVersion = appContext.getVersion();
    String application = "Application: " + applicationName;
    String version = "Version: " + applicationVersion;
    return new String[] { application, version };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <T extends IContext> T getContext(Class<T> contextClass) {
    if (contextClass == null) {
      return null;
    }
    IContext context = this.contexts.get(contextClass);
    if (context != null) {
      try {
        T t = contextClass.cast(context);
        if (!context.isActive()) {
          t.onContextActivate();
        }
        return t;
      }
      catch (Exception e) {
        handleException(e, null);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Locale getLocale() {
    return Runner.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getCharset() {
    return charset;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Charset getServerCharset() {
    return Charset.defaultCharset();
  }
}
