package snakegame;

import csdk.v1_0.helper.application.ApplicationAboutAction;
import csdk.v1_0.helper.application.ApplicationExitAction;
import snakegame.action.ChangeDifficultAction;
import snakegame.action.ChangeMapSizeAction;
import snakegame.action.PlayAction;
import snakegame.algorithm.euclidean.EuclideanAlgorithm;
import snakegame.algorithm.manhattan.ManhattanDistanceAlgorithm;
import snakegame.session.Level;
import snakegame.session.MapSize;
import snakegame.session.Session;

/**
 * Classe responsvel por inicializar as aes usadas na aplicao.
 *
 * @author Tecgraf/PUC-Rio
 */
class Actions {

  /** Ao que inicia o jogo. */
  final PlayAction playAction;

  /** Ao que simula o jogo com o algoritmo guloso 'euclidiano'. */
  final PlayAction greedyAction;

  /** Ao que simula o jogo com o algoritmo guloso 'distncia em manhattan'. */
  final PlayAction manhattanAction;

  /** Ao que modifica o tamanho do mapa para 'pequeno'. */
  final ChangeMapSizeAction smallAction;

  /** Ao que modifica o tamanho do mapa para 'grande'. */
  final ChangeMapSizeAction largeAction;

  /** Ao que modifica o nvel de dificuldade do jogo para 'fcil'. */
  final ChangeDifficultAction easyAction;

  /** Ao que modifica o nvel de dificuldade do jogo para 'normal'. */
  final ChangeDifficultAction normalAction;

  /** Ao que modifica o nvel de dificuldade do jogo para 'difcil'. */
  final ChangeDifficultAction hardAction;

  /** Ao que fecha a aplicao. */
  final ApplicationExitAction exitAction;

  /** Ao que exibe o 'sobre a aplicao'. */
  final ApplicationAboutAction aboutAction;

  /**
   * Construtor.
   *
   * @param application aplicao.
   * @param session sesso.
   */
  Actions(SnakeGame application, Session session) {
    if (application == null) {
      throw new IllegalArgumentException("application no pode ser nulo.");
    }
    if (session == null) {
      throw new IllegalArgumentException("session no pode ser nulo.");
    }

    playAction = new PlayAction(application, session, null);

    greedyAction =
      new PlayAction(application, session, new EuclideanAlgorithm());
    manhattanAction =
      new PlayAction(application, session, new ManhattanDistanceAlgorithm());

    smallAction = new ChangeMapSizeAction(application, session, MapSize.SMALL);
    largeAction = new ChangeMapSizeAction(application, session, MapSize.LARGE);

    easyAction = new ChangeDifficultAction(application, session, Level.EASY);
    normalAction =
      new ChangeDifficultAction(application, session, Level.NORMAL);
    hardAction = new ChangeDifficultAction(application, session, Level.HARD);

    exitAction = new ApplicationExitAction(application);
    aboutAction = new ApplicationAboutAction(application,
      application.getApplicationFrame());
  }
}
