package textviewer;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;

import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JToolBar;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.gui.Task;
import textviewer.actions.CloneAction;
import textviewer.actions.OpenLocalAction;
import textviewer.actions.OpenProjectAction;
import textviewer.actions.SaveAction;
import textviewer.actions.SaveAsLocalAction;
import textviewer.actions.SaveAsProjectAction;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.project.IProjectObserver;
import csdk.v1_0.helper.AbstractCSDKWindowApplication;
import csdk.v1_0.helper.application.ApplicationAboutAction;
import csdk.v1_0.helper.application.ApplicationExitAction;

/**
 * Aplicao de testes mnima do CSDK que permite a abertura e gravao de
 * arquivos ASCII.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class TextViewer extends AbstractCSDKWindowApplication {

  /**
   * Tipo de mensagem que pode ser recebida pela aplicao para definir o texto
   * inicial.
   */
  public static final String TEXT_MESSAGE_TYPE = "text";

  /**
   * Tipo de mensagem que pode ser recebida pela aplicao para definir o
   * arquivo selecionado.
   */
  private static final Object FILE_MESSAGE_TYPE = "file";

  /**
   * rea de texto.
   */
  private final JTextArea textArea;

  /**
   * ltimo arquivo selecionado.
   */
  private IFile selectedFile;

  /**
   * Ao de abrir um arquivo da rea de projetos.
   */
  private final OpenProjectAction openProjectAction;

  /**
   * Ao de salvar um arquivo da rea de projetos.
   */
  private final SaveAsProjectAction saveAsProjectAction;

  /**
   * Construtor.
   * 
   * @param csdkInterface interface padro para o ambiente CSDK.
   */
  public TextViewer(final ICSDKEnvironment csdkInterface) {
    super(csdkInterface);
    this.textArea = new JTextArea();
    this.openProjectAction = new OpenProjectAction(this);
    this.saveAsProjectAction = new SaveAsProjectAction(this);
    this.selectedFile = null;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildAboutMenu() {
    JMenu menu = new JMenu(getString("about.menu"));
    menu.add(new ApplicationAboutAction(this, getApplicationFrame()));
    return menu;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void applicationStarted(JFrame mainFrame) {
    boolean enableProjectActions =
      isProjectContextAvailable() && getCurrentProjectId() != null;
    if (enableProjectActions) {
      IProjectObserver listener = new IProjectObserver() {
        @Override
        public void onProjectOpen(String projectId) {
          setProjectActionsEnabled(true);
        }

        @Override
        public void onProjectClose(String projectId) {
          setProjectActionsEnabled(false);
        }
      };
      this.addProjectObserver(listener);
    }
    setProjectActionsEnabled(enableProjectActions);

    JMenuBar menuBar = buildMenuBar();
    mainFrame.setJMenuBar(menuBar);

    JPanel panel = new JPanel();

    textArea.setFont(new Font(Font.MONOSPACED, Font.BOLD, 14));
    textArea.setTabSize(4);

    JScrollPane scrollPane = new JScrollPane(textArea);
    scrollPane
      .setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
    Dimension scrDim = new Dimension(600, 400);
    scrollPane.setSize(scrDim);
    scrollPane.setPreferredSize(scrDim);
    scrollPane.setMinimumSize(scrDim);

    panel.setLayout(new BorderLayout());
    panel.add(scrollPane, BorderLayout.CENTER);
    panel.add(buildToolbar(), BorderLayout.NORTH);

    mainFrame.getContentPane().add(panel);
    mainFrame.setSize(new Dimension(600, 400));
  }

  /**
   * Montagem do menu.
   * 
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildOptionsMenu());
    menuBar.add(buildAboutMenu());
    return menuBar;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildOptionsMenu() {
    JMenu menu = new JMenu(getString("options.menu"));
    menu.add(openProjectAction);
    menu.add(new OpenLocalAction(this));
    menu.addSeparator();
    menu.add(saveAsProjectAction);
    menu.add(new SaveAsLocalAction(this));
    menu.add(new SaveAction(this));
    menu.addSeparator();
    menu.add(new CloneAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Montagem da toolbar
   * 
   * @return a toolbar
   */
  private JToolBar buildToolbar() {
    JToolBar toolbar = new JToolBar();
    toolbar.add(openProjectAction);
    toolbar.add(saveAsProjectAction);
    toolbar.setFloatable(false);
    return toolbar;
  }

  /**
   * Retorna o texto exibido no programa
   * 
   * @return a string visvel no painel
   */
  public String getText() {
    return textArea.getText();
  }

  /**
   * Ajusta o texto visvel no painel.
   * 
   * @param string o texto
   */
  public void setText(String string) {
    textArea.setText(string);
    textArea.setCaretPosition(0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canEndApplication() {
    if (selectedFile == null) {
      return true;
    }
    else {
      int result =
        JOptionPane.showConfirmDialog(getApplicationFrame(),
          getString("confirm.message"), getApplicationName(),
          JOptionPane.YES_NO_OPTION);
      return result == JOptionPane.YES_OPTION;
    }
  }

  /**
   * Retorna o ltimo arquivo selecionado.
   * 
   * @return selectedFile arquivo selecionado.
   */
  public IFile getSelectedFile() {
    return selectedFile;
  }

  /**
   * Define o arquivo selecionado.
   * 
   * @param selectedFile arquivo selecionado.
   */
  public void setSelectedFile(IFile selectedFile) {
    closeFile(selectedFile);
    this.selectedFile = selectedFile;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessageReceived(IMessage message, IMessageSender sender) {
    String type = message.getType();
    if (type.equals(TEXT_MESSAGE_TYPE)) {
      setText((String) message.getValue());
    }
    if (type.equals(FILE_MESSAGE_TYPE)) {
      IFile value = (IFile) message.getValue();
      openFile(value);
    }
  }

  /**
   * Salva o contedo do visualizador num arquivo.
   * 
   * @param file o arquivo.
   */
  public void saveFile(final IFile file) {
    setSelectedFile(file);
    Task<Void> task = new Task<Void>() {
      @Override
      protected void performTask() throws Exception {
        OutputStreamWriter writer = null;
        try {
          String content = getText();
          OutputStream stream = file.getOutputStream();
          Charset charset = null;
          switch (file.getFileLocationType()) {
            case LOCAL:
              charset = getCharset();
              break;
            case REMOTE:
              charset = getServerCharset();
              break;
            default:
              throw new IllegalStateException("Cannot determine charset");
          }
          writer = new OutputStreamWriter(stream, charset);
          writer.write(content);
          writer.flush();
        }
        finally {
          FileUtils.close(writer);
        }
      }

      @Override
      protected void afterTaskUI() {
        String key = "save.ok.message";
        String msg = getString(key);
        JOptionPane.showMessageDialog(getApplicationFrame(), msg,
          getApplicationName(), JOptionPane.INFORMATION_MESSAGE);
      }
    };
    task.setProgressDialogDelay(0);
    task.execute(getApplicationFrame(), getApplicationName(),
      getString("open.task"));
  }

  /**
   * Abre um arquivo no visualizador.
   * 
   * @param file o arquivo.
   */
  public void openFile(final IFile file) {
    try {
      if (file != null) {
        Task<String> task = new Task<String>() {
          @Override
          protected void performTask() throws Exception {
            InputStreamReader reader = null;
            try {
              StringBuilder stringBuffer = new StringBuilder((int) file.size());
              InputStream in = file.getInputStream();
              Charset charset = null;
              switch (file.getFileLocationType()) {
                case LOCAL:
                  charset = getCharset();
                  break;
                case REMOTE:
                  charset = getServerCharset();
                  break;
                default:
                  throw new IllegalStateException("Cannot determine charset");
              }
              reader = new InputStreamReader(in, charset);
              int BUFFER_SIZE = 128;
              char[] buffer = new char[BUFFER_SIZE];
              int nbytes = reader.read(buffer);
              while (nbytes > 0) {
                stringBuffer.append(buffer, 0, nbytes);
                nbytes = reader.read(buffer);
              }
              setResult(stringBuffer.toString());
            }
            finally {
              FileUtils.close(reader);
            }
          }

          @Override
          protected void afterTaskUI() {
            String text = getResult();
            setText(text);
          }
        };
        task.setProgressDialogDelay(0);
        task.execute(getApplicationFrame(), getApplicationName(),
          getString("open.task"));
      }
    }
    finally {
      setSelectedFile(file);
      closeFile(file);
    }
  }

  /**
   * Define se as aes de projeto devem ser habilitadas.
   * 
   * @param enabled <code>true</code> se as aes devem ser habilitadas ou
   *        <code>false</code>, caso contrrio.
   */
  private void setProjectActionsEnabled(final boolean enabled) {
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        openProjectAction.setEnabled(enabled);
        saveAsProjectAction.setEnabled(enabled);
      }
    });
  }

  /**
   * Fecha o arquivo.
   * 
   * @param file o arquivo.
   */
  private void closeFile(IFile file) {
    try {
      if (file != null) {
        file.close(true);
      }
    }
    catch (IOException e) {
      handleException(e, getApplicationFrame());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void applicationEnded() {
    closeFile(selectedFile);
  }

}
