package tecgraf.javautils.concurrent.locks;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Classe responsvel por observar quando um lock de um objeto  obtido ou
 * expirado.
 */
public class LockFuture implements Remote {

  /**
   * Indica se future  remoto
   */
  private boolean isRemote;

  /**
   * Indica se j chegou a notificao de que o lock foi obtido ou o pedido
   * expirou.
   */
  private boolean gotNotification;

  /**
   * Identificador do lock recebido
   */
  private LockId lockId;

  /**
   * Nmero de tentativas de obteno do lock
   */
  private AtomicInteger numberOfAttempts;

  /**
   * Constri o future que  notificado quando o lock  obtido ou expirado
   * 
   */
  public LockFuture() {
    super();
    this.gotNotification = false;
    this.lockId = null;
    this.numberOfAttempts = new AtomicInteger();
    this.isRemote = false;
  }

  /**
   * Constri o future que  notificado quando o lock  obtido ou expirado e
   * indica se future
   * 
   * @param isRemote flag que indica se future  remoto
   */
  public LockFuture(boolean isRemote) {
    this();
    this.isRemote = isRemote;
    if (this.isRemote) {
      try {
        UnicastRemoteObject.exportObject(this);
      }
      catch (RemoteException ex) {
        //TODO verificar o que fazer caso a exportao do objeto d errado
      }
    }
  }

  /**
   * Mtodo chamado quando o lock  obtido.
   * 
   * @param lockId idenficador do lock obtido
   * @throws RemoteException falha de rmi
   */
  public synchronized final void objectLocked(LockId lockId)
    throws RemoteException {
    this.lockId = lockId;
    gotNotification();
  }

  /**
   * Mtodo chamado quando o lock  expirado por timeout.
   * 
   * @throws RemoteException falha de rmi
   */
  public synchronized final void objectLockExpired() throws RemoteException {
    this.lockId = null;
    gotNotification();
  }

  /**
   * Incrementa o nmero de tentativas de obter o lock
   * 
   * @return nmero de tentativas realizadas
   */
  final int addAttempt() {
    return this.numberOfAttempts.incrementAndGet();
  }

  /**
   * Retorna o nmero de tentativas na obteno do lock
   * 
   * @return o nmero de tentativas na obteno do lock
   */
  public int getNumberOfAttempts() {
    return this.numberOfAttempts.intValue();
  }

  /**
   * Chegou a notificao de que o lock do arquivo foi obtido ou expirado.
   */
  private synchronized void gotNotification() {
    gotNotification = true;
    //verifica se future  remoto
    if (this.isRemote) {
      try {
        UnicastRemoteObject.unexportObject(this, false);
      }
      catch (RemoteException ex) {
        //TODO verificar o que fazer caso a "desexportao" d errado
      }
    }
    notify();
  }

  /**
   * Espera por uma notificao: de <code>objectLocked</code> ou de
   * <code>objectLockExpired</code>.
   * 
   * @return identificador caso lock tenha sido obtido ou <code>null</code> caso
   *         requisio tenha expirado
   */
  public synchronized LockId get() {
    while (!gotNotification) {
      try {
        wait();
      }
      catch (InterruptedException e) {
      }
    }
    return this.lockId;
  }
}
