package tecgraf.javautils.concurrent.locks;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

/**
 * Classe que contm as informaes sobre os locks de um objeto.
 * <ul>
 * <li>a referncia para o objeto
 * <li>a referncia para o <code>LockManager</code>
 * <li>a lista dos locks por poltica
 * <li>o mapeamento do identificador de lock para cada lock
 * </ul>
 * 
 * 
 * @see Lock
 * 
 * @author Tecgraf
 */
class LockInfo {

  /**
   * Objeto referente a este lock
   */
  SharedAccessObject lockedObject;

  /**
   * Flag que informa se so aceitos pedidos de reentrncia
   */
  private boolean allowsReentrance;

  /**
   * Mapa dos locks por poltica
   */
  private HashMap<LockPolicy, Vector<Lock>> lockMap;

  /**
   * Mapa dos locks por id
   */
  private HashMap<LockId, Lock> lockIdMap;

  /**
   * Logger
   */
  private LockLogger logger;

  /**
   * Construtor
   * 
   * @param object objeto com lock
   * @param allowsReentrance Flag que informa se so aceitos pedidos de
   *        reentrncia ou no
   */
  LockInfo(SharedAccessObject object, boolean allowsReentrance) {
    this.lockedObject = object;
    this.allowsReentrance = allowsReentrance;
    this.lockMap = new HashMap<>();
    this.lockIdMap = new HashMap<>();
    this.logger = LockLogger.getInstance();
  }

  /**
   * Retorna a lista contendo todos os locks do objeto
   * 
   * @return a lista de locks de uma determina poltica do objeto
   */
  List<Lock> getLocks() {
    LockPolicy[] policies;
    synchronized (lockMap) {
      policies = this.lockMap.keySet().toArray(new LockPolicy[] {});
      return new ArrayList<>(this.getLocks(null, policies));
    }
  }

  /**
   * Retorna a lista contendo todos os locks do objeto que no tem o mesmo lock
   * de origem
   * 
   * @param originatorLockId identificador do lock origem
   * @return a lista de locks de uma determina poltica do objeto
   */
  List<Lock> getLocks(LockId originatorLockId) {
    LockPolicy[] policies;
    synchronized (lockMap) {
      policies = this.lockMap.keySet().toArray(new LockPolicy[] {});
      return new ArrayList<>(this.getLocks(originatorLockId, policies));
    }
  }

  /**
   * Retorna a lista contendo todos os locks de uma determina poltica do objeto
   * 
   * @param policies polticas do lock
   * 
   * @return a lista de locks de uma determinada poltica do objeto
   */
  Set<Lock> getLocks(LockPolicy... policies) {
    return this.getLocks(null, policies);
  }

  /**
   * Retorna a lista contendo todos os locks de uma determina poltica do objeto
   * que no tem o mesmo identificador de lock de origem
   * 
   * @param originatorLockId identificador do lock origem
   * @param policies polticas do lock
   * 
   * @return a lista de locks de uma determinada poltica do objeto
   */
  Set<Lock> getLocks(LockId originatorLockId, LockPolicy... policies) {
    Set<Lock> locks = new HashSet<>();
    if (policies != null) {
      for (LockPolicy policy : policies) {
        synchronized (lockMap) {
          Vector<Lock> locksList;
          locksList = this.lockMap.get(policy);
          if (locksList != null) {
            for (Lock lock : locksList) {
              //Verifica se identificador origem  diferente de nulo
              if (originatorLockId != null) {
                //Se identificador origem fr diferente do identificador passado como parmetro
                if (!lock.getOriginatorLockId().equals(originatorLockId)) {
                  locks.add(lock);
                }
              }
              else {
                //Se fr passado nulo no identificador origem, no existe esta restrio
                locks.add(lock);
              }
            }
          }
        }
      }
    }
    return locks;
  }

  /**
   * Retorna a lista contendo todos os locks de um determinado usurio para a
   * lista especificada de polticas.
   * 
   * @param ownerKey identificador do usurio
   * @param policies polticas do lock
   * 
   * @return a lista de locks de uma determinada poltica do objeto
   */
  Set<Lock> getLocks(Object ownerKey, LockPolicy... policies) {
    Set<Lock> locks = new HashSet<>();
    for (Lock lock : this.getLocks(policies)) {
      Object lockOwner = lock.getOwnerKey();
      if (lockOwner != null && lockOwner.equals(ownerKey)) {
        locks.add(lock);
      }
    }
    return locks;
  }

  /**
   * Verifica se usurio possui lock de certo tipo
   * 
   * @param policy poltica solicitada
   * @param ownerKey identificao do usurio
   * @return se usurio possui lock de certo tipo
   */
  boolean hasLock(LockPolicy policy, Object ownerKey) {
    LockPolicy lockPolicy = this.getLockPolicy(ownerKey);
    if (lockPolicy == null) {
      return false;
    }
    return policy.getValue() <= lockPolicy.getValue();
  }

  /**
   * Retorna a poltica mais alta de um lock pertencente ao usurio
   * 
   * @param ownerKey
   * @return a poltica mais alta de um lock pertencente ao usurio
   */
  LockPolicy getLockPolicy(Object ownerKey) {
    LockPolicy policy = null;
    for (Lock lock : getLocks()) {
      Object lockOwner = lock.getOwnerKey();
      if (lockOwner != null && lockOwner.equals(ownerKey)) {
        if ((policy == null)
          || (lock.getPolicy().getValue() > policy.getValue())) {
          policy = lock.getPolicy();
        }
      }
    }
    return policy;
  }

  /**
   * Retorna os usurios detentores de locks sobre o objeto
   * 
   * @return o mapa com polticas e conjunto de chaves de usurios detentores de
   *         locks
   */
  Map<LockPolicy, Set<Object>> getLockOwners() {
    LockPolicy[] policies;
    synchronized (lockMap) {
      policies = this.lockMap.keySet().toArray(new LockPolicy[] {});
    }
    return this.getLockOwners(policies);
  }

  /**
   * Retorna os usurios detentores de determinados tipos de locks sobre o
   * objeto.
   * 
   * @param policies polticas do lock
   * 
   * @return mapeamento de politica de lock em conjuntos de usurios que possuem
   *         o lock.
   */
  Map<LockPolicy, Set<Object>> getLockOwners(LockPolicy... policies) {
    Map<LockPolicy, Set<Object>> mapPolicyOwners =
            new HashMap<>();
    if (policies != null) {
      for (LockPolicy policy : policies) {
        Set<Object> ownersByPolicy = new HashSet<>();
        for (Lock lock : this.getLocks(policy)) {
          Object lockOwner = lock.getOwnerKey();
          if (lockOwner != null) {
            ownersByPolicy.add(lockOwner);
          }
        }
        mapPolicyOwners.put(policy, ownersByPolicy);
      }
    }
    return mapPolicyOwners;
  }

  /**
   * Aquisio de um novo lock sobre o objeto
   * 
   * @param lockPolicy poltica de lock solicitada para o objeto
   * @param ownerKey identificador do usurio solicitante do lock
   * @param originatorLockId identificador do lock que originou a criao do
   *        novo lock
   * @return o lock criado
   */
  Lock newLock(LockPolicy lockPolicy, Object ownerKey, LockId originatorLockId) {
    //Cria um novo lock
    Lock theLock = new Lock(lockPolicy, ownerKey, originatorLockId);
    //Recupera a lista de locks da poltica aplicada
    synchronized (lockMap) {
      Vector<Lock> locks;
      locks = lockMap.get(lockPolicy);
      //Se no existem locks para a poltica
      if (locks == null) {
        //Cria a lista de locks para poltica 
        locks = new Vector<>();
      }
      //Insere no mapa de polticas e respectivos locks a nova lista
      this.lockMap.put(lockPolicy, locks);
      //Adiciona o lock criado na lista de locks
      locks.add(theLock);
    }
    //Adiciona o identificador e lock criado no mapa de identificadores
    synchronized (lockIdMap) {
      this.lockIdMap.put(theLock.getId(), theLock);
    }
    //Retorna o lock criado
    return theLock;
  }

  /**
   * Remoo de um lock sobre objeto
   * 
   * @param lockId identificador do lock a ser removido
   * 
   * @return o lock retirado da lista de locks do objeto
   */
  Lock removeLock(LockId lockId) {
    //Recupera lock a partir do identificador
    Lock lock;
    synchronized (lockIdMap) {
      lock = this.lockIdMap.get(lockId);
    }
    //Caso no tenha encontrado
    if (lock == null) {
      logger.fine("Lock no existe no mapeamento entre ids e locks");
      return null;
    }
    //Recupera a lista de locks de acordo com a poltica do lock a ser removido
    synchronized (lockMap) {
      List<Lock> locksByPolicy;
      locksByPolicy = this.lockMap.get(lock.getPolicy());
      //Tenta remover o lock
      if (locksByPolicy != null && locksByPolicy.remove(lock)) {
        //Verifica se pode retirar do mapa a lista de locks pela poltica
        if (locksByPolicy.size() <= 0) {
          //          synchronized (lockMap) {
          this.lockMap.remove(lock.getPolicy());
          //          }
        }
        //Remove a entrada do identificador do mapa de ids de lock
        synchronized (lockIdMap) {
          this.lockIdMap.remove(lockId);
        }
        //Retorna o lock removido
        return lock;
      }
    }
    logger.fine("Lock no existe no mapeamento por polticas");
    return null;
  }

  /**
   * Retorna se  permitido reentrncia em locks ou no
   * 
   * @return true se  permitido reentrncia, caso contrrio, retorna false
   */
  boolean allowsReentrance() {
    return this.allowsReentrance;
  }
}
