package tecgraf.javautils.concurrent.locks;

import java.io.Serializable;
import java.util.Map;
import java.util.Set;

/**
 * Classe que representa o resultado completo de uma aquisio de lock,
 * contendo: (i) indicador se aquisio foi bem sucedida; (ii) identificador do
 * lock caso aquisio tenha sido bem sucedida - caso contrrio, o valor  nulo;
 * (iii) mapa de indetificadores de usurios por lock do objeto alvo da
 * aquisio - este mapa s  preenchido caso aquisio seja negada. *
 * 
 * @author Tecgraf
 */
public class LockResult implements Serializable {

  /**
   * Resultado da tentativa de aquisio
   */
  private boolean acquireResult;

  /**
   * Identificador do pedido de lock
   */
  private LockId lockId;

  /**
   * Mapa com polticas e identificadores dos usurios detentores dos locks
   */
  private Map<LockPolicy, Set<Object>> mapPolicyLockOwners;

  /**
   * Construtor do resultado quando a aquisio de lock obteve sucesso
   * 
   * @param lockId identificador do lock obtido
   */
  public LockResult(LockId lockId) {
    this.acquireResult = true;
    this.lockId = lockId;
  }

  /**
   * Construtor do resultado quando a aquisio do lock falhou
   * 
   * @param mapOwnersByPolicy mapa com identificadores dos usurios por
   *        polticas
   */
  public LockResult(Map<LockPolicy, Set<Object>> mapOwnersByPolicy) {
    this.acquireResult = false;
    this.mapPolicyLockOwners = mapOwnersByPolicy;
  }

  /**
   * Retorna se o resultado da aquisio teve sucesso ou no
   * 
   * @return o resultado da aquisio
   */
  public boolean acquiredLock() {
    return this.acquireResult;
  }

  /**
   * Retorna o identificador do lock se obtido
   * 
   * @return o identificador do lock se obtido
   */
  public LockId getLockId() {
    return this.lockId;
  }

  /**
   * Retorna o mapa com polticas e identificadores dos usurios detentores
   * 
   * @return o mapa com polticas e identificadores dos usurios detentores
   */
  public Map<LockPolicy, Set<Object>> getOwnersByPolicy() {
    return this.mapPolicyLockOwners;
  }

  /**
   * Atribui um mapa com polticas e identificadores de usurios detentores
   * 
   * @param mapPolicyLockOwners o mapa com polticas e identificadores dos
   *        usurios detentores
   */
  public void setOwnersByPolicy(Map<LockPolicy, Set<Object>> mapPolicyLockOwners) {
    this.mapPolicyLockOwners = mapPolicyLockOwners;
  }

}
