package tecgraf.javautils.excel.v1.util;

import java.awt.Component;
import java.awt.Point;

import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.ListModel;
import javax.swing.table.TableModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import tecgraf.javautils.excel.v1.ExcelColor;
import tecgraf.javautils.excel.v1.ExcelConfig;
import tecgraf.javautils.excel.v1.ExcelDataTool;
import tecgraf.javautils.excel.v1.ExcelExportable;
import tecgraf.javautils.excel.v1.ExcelExportableAdapter;
import tecgraf.javautils.excel.v1.ExcelStroke;
import tecgraf.javautils.excel.v1.ExcelStructureTool;
import tecgraf.javautils.excel.v1.ExcelStyle;
import tecgraf.javautils.excel.v1.ExcelStyleTool;
import tecgraf.javautils.excel.v1.ExcelTable;
import tecgraf.javautils.excel.v1.style.DefaultExcelStyleSet;

/**
 * Facilitadores de Excel
 *
 *
 * @author bbreder
 */
public abstract class ExcelTableUtil {

  /**
   * Atribui uma tabela a uma planilha excel
   *
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table) {
    setTable(head, data, style, column, row, table, null);
  }

  /**
   * Atribui uma tabela a uma planilha excel
   *
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   * @param format
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table,
    ExcelTable format) {
    setTable(head, data, style, column, row, table, format, false);
  }

  /**
   * Atribui uma tabela a uma planilha excel
   *
   * @param head
   * @param data
   * @param style
   * @param column
   * @param row
   * @param table
   * @param format
   * @param firstIsRowHeader
   */
  public static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, int column, int row, JTable table, ExcelTable format,
    boolean firstIsRowHeader) {
    if (format == null) {
      throw new IllegalArgumentException("formatador de tabela  nulo");
    }
    data.setCell(column, row);
    TableModel model = table.getModel();
    JScrollPane scroll = getScroll(table);
    if (scroll instanceof ExcelExportable) {
      ExcelExportable excelExportable = (ExcelExportable) scroll;
      excelExportable.exportExcel(head, data, style, format);
    }
    else {
      Point dataPoint = getDataPoint(table, head, data, style);
      int dataRow = row + dataPoint.y;
      int dataColumn = column + dataPoint.x;
      if (scroll != null) {
        showCorner(head, data, style, table, row, column, dataRow, dataColumn,
          format);
        showRowHeader(head, data, style, table, row, column, format);
      }
      showAdvanceColumnHeader(head, data, style, table, row, column, dataRow,
        dataColumn, format);
      exportData(data, style, table, format, firstIsRowHeader, dataRow,
        dataColumn);
      if (firstIsRowHeader) {
        style.addBoxRow(dataColumn, dataRow, dataRow + model.getRowCount() - 1,
          ExcelStroke.MEDIUM, ExcelColor.BLACK);
      }
      if (model.getRowCount() > 0) {
        data.decColumn();
      }
      style.addBox(column, data.getColumn(), row, data.getRow(),
        ExcelStroke.MEDIUM, ExcelColor.BLACK);
      data.incRow();
      data.setColumn(column);
    }
  }

  /**
   * Exporta o contedo da tabela
   *
   * @param data
   * @param style
   * @param table
   * @param format
   * @param firstIsRowHeader
   * @param dataRow
   * @param dataColumn
   */
  public static void exportData(ExcelDataTool data, ExcelStyleTool style,
    JTable table, ExcelTable format, boolean firstIsRowHeader, int dataRow,
    int dataColumn) {
    for (int r = 0; r < table.getRowCount(); r++) {
      data.setCell(dataColumn, dataRow + r);
      for (int c = 0; c < table.getColumnCount(); c++) {
        Object valueAt = table.getValueAt(r, c);
        if (firstIsRowHeader && c == 0) {
          style.setStyle(dataColumn, dataRow + r, DefaultExcelStyleSet
            .buildRowHeader(style));
        }
        if (format != null) {
          valueAt = format.getValue(table, r, c, valueAt);
          ExcelStyle cellStyle = format.getStyle(table, r, c, style);
          if (cellStyle != null) {
            style.setStyle(dataColumn + c, dataRow + r, cellStyle);
          }
        }
        else {
          valueAt = fixTableValue(table, r, c, valueAt);
        }
        data.setCellHorizontal(valueAt);
      }
    }
  }

  /**
   * Modifica o valor de uma celula da tabela
   *
   * @param table
   * @param row
   * @param column
   * @param value
   * @return valor modificado
   */
  private static Object fixTableValue(JTable table, int row, int column,
    Object value) {
    //    if (value instanceof Date) { TODO lbarros
    Component comp = table.getCellRenderer(row, column)
      .getTableCellRendererComponent(table, value, false, false, row, column);
    if (comp instanceof JLabel) {
      value = ((JLabel) comp).getText();
    }
    //    }
    return value;
  }

  /**
   * Imprime o cabealho de linha
   *
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param format
   */
  public static void showRowHeader(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, int row, int column,
    ExcelTable format) {
    JScrollPane scroll = getScroll(table);
    Component rowHeader = getRowHeader(scroll);
    if (rowHeader != null) {
      data.setCell(column, row);
      Component corner = getCorner(scroll);
      if (corner != null) {
        Component columnHeader = getColumnHeader(scroll);
        ExcelExportableAdapter columnAdapter = ExcelConfig.getExportable(
          columnHeader);
        if (columnAdapter != null) {
          columnAdapter.getRowCount(columnHeader);
        }
        else {
          data.incRow();
        }
      }
      ExcelExportableAdapter rowAdapter = ExcelConfig.getExportable(rowHeader);
      if (rowAdapter != null) {
        rowAdapter.exportExcel(rowHeader, head, data, style, format);
      }
      else if (JList.class.isInstance(rowHeader)) {
        JList list = (JList) rowHeader;
        ListModel rowModel = list.getModel();
        for (int r = 0; r < rowModel.getSize(); r++) {
          Object valueAt = rowModel.getElementAt(r);
          ExcelStyle cellStyle = null;
          if (format != null) {
            valueAt = format.getRowHeaderValue(table, list, r, valueAt);
            cellStyle = format.getRowHeaderStyle(table, list, row + r, style);
          }
          if (cellStyle == null) {
            cellStyle = DefaultExcelStyleSet.buildRowHeader(style);
          }
          style.setStyle(data.getColumn(), data.getRow(), cellStyle);
          data.setCellVertical(valueAt);
        }
        style.addBox(column, column, row, row + rowModel.getSize(),
          ExcelStroke.MEDIUM, ExcelColor.BLACK);
      }
      else if (JTree.class.isInstance(rowHeader)) {
        JTree tree = (JTree) rowHeader;
        TreeModel rowModel = tree.getModel();
        if (TreeNode.class.isInstance(rowModel.getRoot())) {
          TreeNode node = (TreeNode) rowModel.getRoot();
          setTable(head, data, style, table, tree, node, 0, format);
          style.addBox(column, column, row, data.getRow() - 1,
            ExcelStroke.MEDIUM, ExcelColor.BLACK);
          style.addBox(column, column + table.getColumnCount(), row + 1, data
            .getRow() - 1, ExcelStroke.MEDIUM, ExcelColor.BLACK);
        }
      }
    }
  }

  /**
   * Retorna o cabealho de linha
   * 
   * @param scroll
   * @return cabealho de linha
   */
  protected static Component getRowHeader(JScrollPane scroll) {
    boolean hasRowHeader = scroll.getRowHeader() != null && scroll
      .getRowHeader().getView() != null;
    if (!hasRowHeader) {
      return null;
    }
    return hasRowHeader ? scroll.getRowHeader().getView() : null;
  }

  /**
   * Imprime o cabealho de coluna no modo avanado
   *
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param dataRow
   * @param dataColumn
   * @param format
   */
  private static void showAdvanceColumnHeader(ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style, JTable table, int row, int column,
    int dataRow, int dataColumn, ExcelTable format) {
    Component columnHeader = getColumnHeader(getScroll(table));
    ExcelExportableAdapter exportableAdapter = ExcelConfig.getExportable(
      columnHeader);
    if (columnHeader instanceof ExcelExportable) {
      ExcelExportable excelExportable = (ExcelExportable) columnHeader;
      data.setCell(column, row);
      excelExportable.exportExcel(head, data, style, format);
    }
    else if (exportableAdapter != null) {
      data.setCell(column, row);
      exportableAdapter.exportExcel(columnHeader, head, data, style, format);
    }
    else {
      showBasicColumnHeader(head, data, style, table, dataRow, dataColumn,
        format);
    }
  }

  /**
   * Retorna o cabealho de coluna
   * 
   * @param scroll
   * @return cabealho de coluna
   */
  protected static Component getColumnHeader(JScrollPane scroll) {
    if (scroll == null || scroll.getColumnHeader() == null) {
      return null;
    }
    return scroll.getColumnHeader().getView();
  }

  /**
   * Imprime o cabealho de linha no modo basico
   *
   * @param head
   * @param data
   * @param style
   * @param table
   * @param dataRow
   * @param dataColumn
   * @param format
   */
  public static void showBasicColumnHeader(ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style, JTable table, int dataRow,
    int dataColumn, ExcelTable format) {
    int row = dataRow - 1;
    int column = dataColumn;
    // Modelo da tabela
    TableModel model = table.getModel();
    // Escreve o cabealho de coluna
    data.setCell(column, row);

    for (int c = 0; c < model.getColumnCount(); c++) {
      String valueAt = model.getColumnName(c);
      if (valueAt != null) {
        valueAt = valueAt.replace("\n", " ");
      }
      else {
        valueAt = " ";
      }
      if (format != null) {
        valueAt = format.getColumnHeaderValue(table, c, valueAt).toString();
        ExcelStyle cellStyle = format.getColumnHeaderStyle(table, column + c,
          style);
        if (cellStyle != null) {
          style.setStyle(column + c, row, cellStyle);
        }
        else {
          style.setStyle(column + c, row, DefaultExcelStyleSet
            .buildColumnHeader(style));
        }
      }
      else {
        style.setStyle(column + c, row, DefaultExcelStyleSet.buildColumnHeader(
          style));
      }
      data.setCell(column + c, row, valueAt);
    }
    style.addBoxColumn(column, column + model.getColumnCount() - 1, row,
      ExcelStroke.MEDIUM, ExcelColor.BLACK);
  }

  /**
   * Retorna o quanto de linhas e colunas o cabealho da tabela est ocupando
   *
   * @param table
   * @param data
   * @param head
   * @param style
   * @return ponto
   */
  private static Point getDataPoint(JTable table, ExcelStructureTool head,
    ExcelDataTool data, ExcelStyleTool style) {
    JScrollPane scroll = getScroll(table);
    if (scroll == null && table.getColumnCount() > 0) {
      return new Point(0, 1);
    }
    boolean hasRowHeader = scroll.getRowHeader() != null && scroll
      .getRowHeader().getView() != null;
    Component rowHeader = null;
    if (hasRowHeader) {
      scroll.getRowHeader().getView();
    }
    boolean hasColumnHeader = scroll.getColumnHeader() != null && scroll
      .getColumnHeader().getView() != null;
    int rows = 1;
    int columns = 0;
    if (hasRowHeader || hasCorner(scroll)) {
      columns = 1;
      if (JList.class.isInstance(rowHeader)) {
        columns = 1;
      }
      else if (JTree.class.isInstance(rowHeader)) {
        columns = 1;
      }
    }
    if (hasColumnHeader) {
      Component columnHeader = scroll.getColumnHeader().getView();
      ExcelExportableAdapter adapter = ExcelConfig.getExportable(columnHeader);
      if (adapter != null) {
        rows = adapter.getRowCount(columnHeader);
      }
    }
    return new Point(columns, rows);
  }

  /**
   * Funo responsvel por exibir o corner da tabela
   *
   * @param head
   * @param data
   * @param style
   * @param table
   * @param row
   * @param column
   * @param dataColumn
   * @param dataRow
   * @param format
   */
  public static void showCorner(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, int row, int column, int dataRow,
    int dataColumn, ExcelTable format) {
    style.setStyle(column, row, DefaultExcelStyleSet.buildColumnHeader(style));
    style.addBox(column, row, ExcelStroke.MEDIUM, ExcelColor.BLACK);
    Component corner = getCorner(getScroll(table));
    if (corner != null && format != null) {
      Object value = format.getCorner(corner);
      if (value != null) {
        data.setCell(column, row, value.toString().replaceAll("<.*?>", ""));
      }
    }
    if (dataRow - row > 1 || dataColumn - column > 1) {
      head.merge(column, dataColumn, row, dataRow - 1);
    }
  }

  /**
   * Funo responsvel por exibir o corner da tabela
   *
   * @param scroll
   * @return tem corner
   */
  private static boolean hasCorner(JScrollPane scroll) {
    return scroll.getCorner(JScrollPane.UPPER_LEFT_CORNER) != null;
  }

  /**
   * Escreve os ns de uma arvore, no cabealho de linha.
   *
   * @param head
   * @param data
   * @param style
   * @param table
   * @param tree
   * @param node
   * @param deep
   * @param format
   */
  private static void setTable(ExcelStructureTool head, ExcelDataTool data,
    ExcelStyleTool style, JTable table, JTree tree, TreeNode node, int deep,
    ExcelTable format) {

    for (int r = 0; r < node.getChildCount(); r++) {
      TreeNode child = node.getChildAt(r);
      String text = child.toString().replace('\n', ' ');
      if (format != null) {
        text = format.getRowHeaderValue(table, tree, r, text).toString();
      }
      for (int n = 0; n < deep; n++) {
        text = "     " + text;
      }
      data.setCellVertical(text);
      int row = data.getRow() - 1;
      style.setStyle(data.getColumn(), row, DefaultExcelStyleSet.buildRowHeader(
        style));
      if (format != null) {
        ExcelStyle cellStyle = format.getRowHeaderStyle(table, tree, row,
          style);
        if (cellStyle != null) {
          style.setStyle(data.getColumn(), row, cellStyle);
        }
        else {
          style.setStyle(data.getColumn(), row, DefaultExcelStyleSet
            .buildRowHeader(style));
        }
      }
      else {
        style.setStyle(data.getColumn(), row, DefaultExcelStyleSet
          .buildRowHeader(style));
      }
      TreePath path = buildTreePath(child);
      if (child.getChildCount() > 0 && tree.isExpanded(path) && deep < data
        .getMaxRowDeep()) {
        data.incColumn();
        data.decColumn();
        setTable(head, data, style, table, tree, child, deep + 1, format);
      }
    }
  }

  /**
   * Constroi o TreePath correspondente ao n
   *
   * @param child
   * @return TreePath do n
   */
  private static TreePath buildTreePath(TreeNode child) {
    if (child.getParent() != null) {
      TreePath parent = buildTreePath(child.getParent());
      int size = parent.getPath().length;
      Object[] paths = new Object[size + 1];
      System.arraycopy(parent.getPath(), 0, paths, 0, size);
      paths[size] = child;
      return new TreePath(paths);
    }
    else {
      return new TreePath(child);
    }
  }

  /**
   * Retorna o scroll da tabela
   *
   * @param table
   * @return scroll ou nulo
   */
  private static JScrollPane getScroll(JTable table) {
    if (table.getParent() != null && JViewport.class.isInstance(table
      .getParent()) && table.getParent().getParent() != null
      && JScrollPane.class.isInstance(table.getParent().getParent())) {
      return (JScrollPane) table.getParent().getParent();
    }
    else {
      return null;
    }
  }

  /**
   * Retorna o Corner do ScrollPane
   *
   * @param scroll
   * @return corner
   */
  public static Component getCorner(JScrollPane scroll) {
    return scroll.getCorner(JScrollPane.UPPER_LEFT_CORNER);
  }

}
