package tecgraf.javautils.gui.field;

import javax.swing.text.BadLocationException;
import javax.swing.text.Document;

/**
 * Campo genrico para mscara Regex. O construtor "GenericField(String regex)"
 * recebe uma expresso regular para definir os caracteres vlidos e a sequncia
 * permitida para o campo. Essa expresso permitir ou no que o texto digitado
 * seja aceito. A cada entrada, o texto atual acrescentado da nova entrada ser
 * verificado na expresso regular. Vale lembrar que o valor do campo dever ser
 * validado pelo mtodo isValidValue e no por essa expresso regular. Para um
 * campo do tipo NN/N, definido por uma expresso "[0-9]{1,2}/[0-9]", esse
 * mtodo dever retornar uma expresso com grupos no obrigatrios:
 * "([0-9]{1,2})?(/)?([0-9])?", pois dessa maneira permitir que cada caracter
 * da sequncia seja digitado separadamente ou omitindo alguns podendo ser
 * corrigido futuramente. Uma segunda opo seria: "[0-9]{1,2}(/([0-9])?)?" o
 * que tambm permite que cada caracter da sequncia seja digitado separadamente
 * porm obrigatoriamente na ordem NN -> NN/ -> NN/N. Esse campo no valida o
 * valor do campo, o mtodo isValidValue sempre retornar True se o mesmo no
 * for sobrescrito. O construtor
 * "GenericField(String regex, String validateRegex)" recebe duas expresses
 * regulares. Respectivamente, a primeira serve para aceitao de quais
 * caracteres devero ser bloqueados ou adicionados ao texto, a segunda  usada
 * para validao da expresso digitada no campo. Ou seja, utilizando este
 * construtor ser possvel validar o valor deste campo baseado na expresso
 * regular definida.
 */
public class GenericField extends AbstractRegexField {

  /**
   * Constri um campo genrico.
   * 
   * @param regex uma expresso regular que define o que pode ser escrito no
   *        campo de texto.
   */
  public GenericField(String regex) {
    this(regex, null);
  }

  /**
   * Constri um campo genrico.
   * 
   * @param regex uma expresso regular que define o que pode ser escrito no
   *        campo de texto.
   * @param validateRegex uma expresso regular usada na validao deste campo
   *        genrico.
   */
  public GenericField(String regex, String validateRegex) {   
    setDocument(new GenericDocument(regex, validateRegex));    
  }


  /**
   * Valida este campo genrico. Se este campo possui uma expresso regular para
   * validao ela  utilizada, seno, a expresso sempre ser vlida.
   * 
   * @return true se o texto encontrado no campo  vlido, false seno.
   */
  @Override
  public boolean isValidValue() {
    GenericDocument document = (GenericDocument) getDocument();
    return document.isValid();
  }

  @Override
  public void setDocument(Document doc) {
    if (doc != null && doc instanceof GenericDocument) {
      super.setDocument(doc);
    }
  }

  private static class GenericDocument extends RegexDocument {

    private String validateRegex;

    public GenericDocument(String regex, String validateRegex) {
      super(regex);
      this.validateRegex = validateRegex;
    }

    @Override
    protected Object getValue() {
      try {
        return getText(0, getLength());
      }
      catch (BadLocationException e) {
        e.printStackTrace();
      }
      return null;
    }

    public boolean isValid() {
      if (validateRegex == null) {
        return true;
      }
      String text = (String) getValue();
      if (text != null)
        return text.matches(validateRegex);
      return false;
    }

  }
}
