package tecgraf.javautils.gui;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import javax.swing.DefaultComboBoxModel;

/**
 * Modelo que garante que os itens so armazenados em uma ordem de
 * classificao. O padro  classificar na ordem natural do item, mas um
 * comparador pode ser usado para personalizar a ordem de classificao.
 * 
 * Cada atualizao no modelo far com que o item seja inserido na ordem de
 * classificao.
 * 
 * @param <T> Um objeto comparable.
 * 
 * @author Tecgraf
 */
public class SortedComboBoxModel<T extends Comparable<T>> extends
  DefaultComboBoxModel {

  /** Comparator para ordenao do combobox */
  private Comparator<T> comparator;

  /**
   * Indica se haver elemento que no sofrer ordenao, que ficar sempre na
   * primeira posio do modelo. Este elemento deve ser inserido na primeira
   * posio do array. O default  no haver elemento especial.
   */
  private boolean hasSpecialElement = false;

  /**
   * Construtor. Cria um modelo vazio que usar a ordenao natural dos itens.
   */
  public SortedComboBoxModel() {
  }

  /**
   * Construtor. Cria um modelo vazio que usar a ordenao natural dos itens.
   * 
   * @param hasSpecialElement Indica se haver um elemento especial que ficar
   *        not topo da lista e no ser passivel de ordenao.
   */
  public SortedComboBoxModel(boolean hasSpecialElement) {
    this.hasSpecialElement = hasSpecialElement;
  }

  /**
   * Construtor. Cria um modelo com dados que usar a ordenao natural dos
   * itens.
   * 
   * @param items Itens que sero adicionados no modelo.
   */
  public SortedComboBoxModel(T[] items) {
    this(items, null);
  }

  /**
   * Construtor. Cria um modelo vazio e usa um comparador especfico.
   * 
   * @param comparator Um comparador especial para ordenao do modelo.
   */
  public SortedComboBoxModel(Comparator<T> comparator) {
    this(null, comparator);
    this.comparator = comparator;
  }

  /**
   * Construtor. Cria um modelo com dados e usa um comparador especfico.
   * 
   * @param items Itens que sero adicionados no modelo.
   * @param comparator Um comparador especial para ordenao do modelo.
   */

  public SortedComboBoxModel(T[] items, Comparator<T> comparator) {
    super();
    this.comparator = comparator;
    if (items != null) {
      for (int i = 0; i < items.length; i++) {
        addElement(items[i]);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addElement(Object element) {
    insertElementAt(element, 0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void insertElementAt(Object element, int index) {
    if (!(element instanceof Comparable<?>)) {
      throw new IllegalArgumentException(
        "Parmetro deve implementar Comparable.");
    }

    int size = getSize();
    int initial = (hasSpecialElement) ? 1 : 0;
    //Determina onde o elemento ser inserido para manter o modelo ordenado
    for (index = initial; index < size; index++) {
      if (comparator != null) {
        T object = (T) getElementAt(index);

        if (comparator.compare(object, (T) element) > 0) {
          break;
        }
      }
      else {
        Comparable<Object> comparable =
          (Comparable<Object>) getElementAt(index);

        if (comparable.compareTo(element) > 0) {
          break;
        }
      }
    }

    //tratamento para o caso de (hasEspecialElement == false) e for o primeiro elemento do modelo.
    index = (size == 0) ? 0 : index;
    super.insertElementAt(element, index);
  }

  /**
   * Mtodo de teste.
   * 
   * @param args
   */
  public static void main(String[] args) {
    SortedComboBoxModel<String> model =
      new SortedComboBoxModel<String>(new String[] { "5", "7", "2", "1", "0" });
    model.addElement("3");
    model.addElement("4");
    model.addElement("8");
    int size = model.getSize();
    List<String> list = new ArrayList<String>();
    for (int n = 0; n < size; n++) {
      list.add((String) model.getElementAt(n));
    }
    System.out.println(list.toString());
  }

}
