/*
 * $Id$
 */
package tecgraf.javautils.gui.print;

import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.print.PageFormat;

/**
 * Texto a ser impresso. O texto pode possuir um ttulo que fica  esquerda. Se
 * houver um ttulo, o texto fica alinhado depois do ttulo.<br>
 * Por exemplo:
 * 
 * <pre>
 * ttulo: xxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
 *         xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
 *         xxxxxxxxxxxxxxxxxxxxxxxxxxxx
 * </pre>
 */
public class PrintableText implements PrintableReportItem {
  /** Alinhamento central */
  public static final String ALIGN_CENTER = TextTool.ALIGN_CENTER;

  /** Ttulo da rea */
  private String title;
  /** Fonte do ttulo */
  private Font titleFont;
  /** Texto da rea */
  private String text;
  /** Fonte do texto */
  private Font textFont;
  /** Altura do componente impresso na pgina */
  private float height;
  /** Pgina corrente sendo impressa */
  private int currentPage;
  /** Impresso em andamento? */
  private boolean printing;
  /** Simulao em andamento? */
  private boolean simulating;
  /** Alinhamento do texto */
  private String alignment;
  /** Ferramenta de impresso de texto */
  protected TextTool textTool;
  /** Retngulo para uso temporrio */
  protected Rectangle2D rect;
  /** Ponto para uso temporrio */
  protected Point2D pt;
  /** Posio inicial do texto a ser escrito */
  private int textPos;
  /** Posio inicial antiga do texto a ser escrito */
  private int oldTextPos;
  /** Espaamento do texto  esquerda */
  private float leftInset;
  /** Indica se a fonte pode ser alterada pela configurao */
  private boolean changeFont;
  /** Largura do texto a ser impresso */
  private float width;

  /**
   * Construtor de textos sem ttulo.
   * 
   * @param text texto da rea.
   * @param textFont fonte do texto.
   */
  public PrintableText(String text, Font textFont) {
    this(null, null, text, textFont, true);
  }

  /**
   * Construtor de textos sem ttulo.
   * 
   * @param text texto da rea.
   * @param textFont fonte do texto.
   * @param changeFont indica se a fonte pode ser alterada pela configurao.
   */
  public PrintableText(String text, Font textFont, boolean changeFont) {
    this(null, null, text, textFont, changeFont);
  }

  /**
   * Construtor de textos com ttulo.
   * 
   * @param title ttulo.
   * @param titleFont fonte do ttulo.
   * @param text texto.
   * @param textFont fonte do texto.
   */
  public PrintableText(String title, Font titleFont, String text, Font textFont) {
    this(title, titleFont, text, textFont, true);
  }

  /**
   * Construtor de textos com ttulo.
   * 
   * @param title ttulo.
   * @param titleFont fonte do ttulo.
   * @param text texto.
   * @param textFont fonte do texto.
   * @param changeFont indica se a fonte pode ser alterada pela proporo.
   */
  public PrintableText(String title, Font titleFont, String text,
    Font textFont, boolean changeFont) {
    this.title = title;
    this.titleFont = titleFont;
    this.text = text;
    this.alignment = TextTool.ALIGN_LEFT;
    this.textFont = textFont;
    this.changeFont = changeFont;
    this.textTool = new TextTool();
    this.rect = new Rectangle2D.Float();
    this.pt = new Point2D.Float();
    this.leftInset = 0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean simulatePrint(Graphics2D g2, PageFormat format, int pageIndex) {
    if (!simulating) {
      init(format);
      simulating = true;
    }
    return print(g2, format, pageIndex, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean print(Graphics2D g2, PageFormat format, int pageIndex) {
    if (!printing) {
      init(format);
      printing = true;
    }
    return print(g2, format, pageIndex, true);
  }

  /**
   * Inicia a impresso.
   * 
   * @param format formato da pgina.
   */
  private void init(PageFormat format) {
    this.currentPage = 0;
    this.textPos = 0;
    this.oldTextPos = 0;
  }

  /**
   * Imprime o texto se o parmetro draw for verdadeiro, seno apenas calcula a
   * altura do texto impresso.
   * 
   * @param g2 contexto grfico.
   * @param format formato da pgina
   * @param pageIndex ndice da pgina.
   * @param draw indica se o texto deve ser impresso.
   * 
   * @return verdadeiro se o texto foi todo impresso.
   */
  private boolean print(Graphics2D g2, PageFormat format, int pageIndex,
    boolean draw) {
    if (pageIndex == currentPage) {
      textPos = oldTextPos;
    }
    currentPage = pageIndex;
    oldTextPos = textPos;
    float titleWidth = 0;
    if (title != null) {
      g2.setFont(titleFont);
      pt.setLocation(format.getImageableX() + leftInset, format.getImageableY());
      textTool.getBBox(g2, title, pt, TextTool.NORTH_WEST, rect);
      if (rect.getHeight() > format.getImageableHeight()) {
        return false;
      }
      if (draw) {
        textTool.draw(g2, title, pt, TextTool.NORTH_WEST, TextTool.ALIGN_LEFT);
      }
      titleWidth = (float) rect.getWidth() + 5;
    }

    return printText(g2, format, titleWidth, draw);
  }

  /**
   * Imprime o texto, se o parmetro draw for verdadeiro. Seno apenas calcula a
   * altura do texto.
   * 
   * @param g2 componente grfico.
   * @param pageFormat formato da pgina.
   * @param titleWidth largura do ttulo j impresso.
   * @param draw indica se o texto deve ser impresso.
   * 
   * @return verdadeiro se o texto foi totalmente impresso na pgina.
   */
  private boolean printText(Graphics2D g2, PageFormat pageFormat,
    float titleWidth, boolean draw) {
    g2.setFont(textFont);

    float x = (float) pageFormat.getImageableX() + titleWidth + leftInset;
    float y = (float) pageFormat.getImageableY();

    float pageWidth = (float) pageFormat.getImageableWidth() - titleWidth;
    float pageHeight = (float) pageFormat.getImageableHeight();
    rect.setRect(x, y, pageWidth, pageHeight);

    if (draw) {
      textPos =
        textTool.draw(g2, text, textPos, rect, TextTool.WRAP_WORD, alignment);
    }
    else {
      textPos =
        textTool
          .getBBox(g2, text, textPos, rect, TextTool.WRAP_WORD, alignment);
    }
    height = (float) rect.getHeight();
    width = textTool.getMaxWidth();

    boolean fullPrinted = textPos >= text.length();
    if (fullPrinted) {
      textPos = 0;
    }
    return fullPrinted;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getHeight() {
    return height;
  }

  /**
   * Obtm o texto da rea.
   * 
   * @return texto da rea.
   */
  public String getText() {
    return this.text;
  }

  /**
   * Obtm o ttulo do texto.
   * 
   * @return ttulo do texto.
   */
  public String getTitle() {
    return this.title;
  }

  /**
   * Indica o alinhamento do texto.
   * 
   * @param alignment alinhamento do texto.
   */
  public void setAlignment(String alignment) {
    this.alignment = alignment;
  }

  /**
   * Indica o espaamento do texto  esquerda em centmetros.
   * 
   * @param leftInset espaamento  esquerda em centmetros.
   */
  public void setLeftInset(float leftInset) {
    this.leftInset = (float) (leftInset * CM_TO_INCH);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initPrinting(PrintConfiguration configuration) {
    printing = false;
    simulating = false;
    if (!changeFont) {
      return;
    }
    textFont =
      textFont.deriveFont(textFont.getSize() * configuration.getFontRate());
    if (titleFont != null) {
      titleFont =
        titleFont.deriveFont(titleFont.getSize() * configuration.getFontRate());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public float getWidth() {
    return width;
  }
}
