package tecgraf.javautils.gui.searchpanel;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;

/**
 * Um painel de busca em componente do tipo <code>AbstractSearchView</code>.
 * 
 * H implementaes conhecidas para AbstractSearchView usando JTree (Bandeira),
 * JList (Planref) e JTable (Bandeira).
 * 
 * @author Tecgraf
 */
public class SearchPanel extends JPanel {

  /** Constante para identificar o campo de busca. */
  public static final String SEARCH_FIELD = "SEARCH_FIELD";
  /** O componente usado na consulta. Pode ser uma rvore ou uma lista. */
  private AbstractSearchView searchComponent;
  /** Define se  desejvel que o campo de busca seja limpo ao perder o foco. */
  private boolean isFocusLostDesirable;

  /**
   * Construtor.
   * 
   * @param searchComponent Componente usado na consulta.
   * @param isFocusLostDesirable Define se  desejvel que o campo de busca seja
   *        limpo ao perder o foco.
   */
  public SearchPanel(AbstractSearchView searchComponent,
    boolean isFocusLostDesirable) {
    this.searchComponent = searchComponent;
    this.isFocusLostDesirable = isFocusLostDesirable;
    build();
  }

  /**
   * Construtor.
   * 
   * @param searchComponent Componente usado na consulta.
   */
  public SearchPanel(AbstractSearchView searchComponent) {
    this(searchComponent, true);
  }

  /**
   * Constri o contedo do painel.
   */
  private void build() {
    GridBagLayout gridBag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    this.setLayout(gridBag);
    // Label de Busca
    JLabel searchLabel =
      new JLabel(LNG.get("tecgraf.javautils.gui.searchpanel.label"));
    searchLabel.setDisplayedMnemonic(LNG.get(
      "tecgraf.javautils.gui.searchpanel.label.mnemonic").charAt(0));
    c.insets = new Insets(5, 5, 0, 5);
    c.gridx = 0;
    c.gridy = 0;
    c.weightx = 0;
    c.weighty = 0;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.WEST;
    gridBag.setConstraints(searchLabel, c);
    this.add(searchLabel);
    // Campo de entrada de texto de busca
    JTextField searchField = makeSearchField();
    searchField.setName(SEARCH_FIELD);
    searchLabel.setLabelFor(searchField);
    c.insets = new Insets(5, 5, 0, 5);
    c.gridx = 1;
    c.gridy = 0;
    c.weightx = 1;
    c.weighty = 0;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.fill = GridBagConstraints.HORIZONTAL;
    gridBag.setConstraints(searchField, c);
    this.add(searchField);
  }

  /**
   * Constri o campo de entrada de texto para a busca.
   * 
   * @return o campo de entrada.
   */
  private JTextField makeSearchField() {
    final JTextField searchField = new JTextField(12);
    searchField.addKeyListener(new KeyAdapter() {
      @Override
      public void keyReleased(KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_F3) {
          SearchPanel.this.searchComponent.selectSearchNext(searchField
            .getText().toUpperCase());
        }
        else {
          SearchPanel.this.searchComponent.selectSearch(searchField.getText()
            .toUpperCase(), false);
        }
      }
    });

    if (isFocusLostDesirable) {
      searchField.addFocusListener(new FocusAdapter() {
        @Override
        public void focusLost(FocusEvent e) {
          searchField.setText("");
        }
      });
    }
    return searchField;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(final boolean enabled) {
    for (Component com : getComponents()) {
      com.setEnabled(enabled);
    }
  }
}
