package tecgraf.javautils.gui.selector;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableRowSorter;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * A classe <code>ItemContainer</code> modela um conjunto de elementos, usando
 * uma tabela ordenvel<code>SortableTable</code>. A formatao da tabela
 * obedece ao formatador que devolve as colunas de cabealho bem como os valores
 * apresentados em cada linha. O conjunto representado por
 * <code>ItemContainer</code> possui dois botes associados:
 * <ul>
 * <li>um que permite remover todos os elementos do conjunto;</li>
 * <li>outro que permite remover elementos selecionados do conjunto.</li>
 * </ul>
 * 
 * @param <T> .
 */
public class ItemContainer<T> {
  /** A tabela ordenvel que representa o ItemContainer. */
  private SortableTable container;

  /** O modelo da tabela que representa o ItemContainer. */
  private ObjectTableModel<T> itemContainerModel;

  /** O componente de interface que representa o conjunto. */
  private JScrollPane view;

  /** O boto que permite operar sobre elementos selecionados do conjunto. */
  private JButton removeSelectedButton;

  /** O boto que permite operar com todos os elementos do conjunto; */
  private JButton removeAllButton;

  /** Indica se o container est habilitado ou no. */
  private boolean enabled;

  /** Lista de listeners */
  private List<ItemContainerListener> itemListeners;

  /**
   * Flag que indica se a tabela deve ser ajustada para ocupar, pelo menos, o
   * tamanho mnimo para que no seja necessrio redimensionar suas colunas para
   * que seja possvel ler todo o contedo destas.
   */
  private boolean adjustToMinimumWidth;

  /** Tamanho mnimo da tabela. */
  private Dimension minimunSize = new Dimension(0, 0);

  /** Tamanho mnimo das colunas da tabela. */
  private ArrayList<Integer> colunsMinimunSize;

  /** Flag que indica se  necessrio reajustar o tamanho mnimo das colunas. */
  private boolean isPendingColumnAdjustment = false;

  /** row sorter responsvel por filtrar os elementos da tabela, se necessrio */
  private TableRowSorter<ObjectTableModel> trs;

  /**
   * Devolve o componente de interface que modela o conjunto.
   * 
   * @return Uma tabela.
   */
  JComponent getView() {
    return view;
  }

  /**
   * Devolve todos os elementos do conjunto.
   * 
   * @return Os elementos do conjunto.
   */
  List<T> getAll() {
    return itemContainerModel.getRows();
  }

  /**
   * Carrega a tabela com elementos. Caso esta esteja ordenada por alguma
   * coluna, os novos elementos so inseridos respeitando este critrio de
   * ordenao. Os elementos so inseridos como linhas da tabela.
   * 
   * @param elements O array de elementos a serem carregados.
   */
  void load(Collection<T> elements) {
    if (elements == null) {
      throw new IllegalArgumentException("elements == null");
    }
    int countInserted = elements.size();
    itemContainerModel.addAll(elements);
    itemContainerModel.fireTableDataChanged();
    if (countInserted == 0) {
      removeAllButton.setEnabled(false);
    }
    else {
      removeAllButton.setEnabled(enabled);
    }

    updateTableAdjustment();

  }

  /**
   * Remove os elementos selecionados no conjunto. As respectivas linhas da
   * tabela so removidas.
   * 
   * @return Os elementos removidos do conjunto.
   */
  Collection<T> removeSelectedElements() {
    int[] viewRowIndexes = container.getSelectedRows();
    int viewRowIndexesLength = viewRowIndexes.length;
    int[] modelRowIndexes = new int[viewRowIndexesLength];
    for (int i = 0; i < viewRowIndexesLength; i++) {
      modelRowIndexes[i] = container.convertRowIndexToModel(viewRowIndexes[i]);
    }
    Collection<T> status = itemContainerModel.removeAll(modelRowIndexes);

    updateTableAdjustment();
    return status;
  }

  /**
   * Remove todos os elementos do conjunto. As respectivas linhas da tabela so
   * removidas.
   * 
   * @return Os elementos removidos.
   */
  Collection<T> removeAll() {
    removeSelectedButton.setEnabled(false);
    removeAllButton.setEnabled(false);
    Collection<T> status = itemContainerModel.removeAll();

    updateTableAdjustment();
    return status;
  }

  /**
   * Atualiza o tamanho mnimo das colunas da tabela.
   */
  private void updateTableAdjustment() {
    if (adjustToMinimumWidth) {
      updateMinimunSize();
    }
    isPendingColumnAdjustment = true;
    ItemContainer.this.view.invalidate();
  }

  /**
   * Verifica se existe algum item selecionado no container.
   * 
   * @return <code>true</code> caso exista algum item selecionado, ou
   *         <code>false</code> caso contrrio.
   */
  boolean hasSelectedItems() {
    ListSelectionModel listModel = container.getSelectionModel();
    return (listModel.getMinSelectionIndex() != -1);
  }

  /**
   * Adiciona um ItemContainerListener
   * 
   * @param listener ItemContainerListener
   * 
   * @throws IllegalArgumentException se o listener estiver null
   */
  public void addItemContainerListener(ItemContainerListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("Listener = null == Illegal ");
    }
    this.itemListeners.add(listener);
  }

  /**
   * Remove um ItemContainerListener
   * 
   * @param listener ItemContainerListener
   * 
   * @return boolean se o listener foi removido
   * 
   * @throws IllegalArgumentException se o listener estiver null
   */
  public boolean removeItemContainerListener(ItemContainerListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("Listener = null == Illegal ");
    }
    return this.itemListeners.remove(listener);
  }

  /**
   * Habilita ou desabilita esse container.
   * 
   * Os botes que permite operar sobre todos os elementos do conjunto e sobre
   * os elementos selecionados no conjunto tambm devem mudar seu estado. Se for
   * para habilitar, os estados dos botes sero analisados de acordo com outras
   * condies sobre a lista de elementos do container. Se for para desabilitar
   * o container, ento os botes devem ser desabilitados.
   * 
   * @param enabled
   */
  void setEnabled(boolean enabled) {
    this.enabled = enabled;
    this.container.setEnabled(enabled);
    if (!enabled) {
      removeAllButton.setEnabled(false);
      removeSelectedButton.setEnabled(false);
    }
  }

  /**
   * Obtm a SortableTable.
   * 
   * @return A SortableTable do ItemContainer
   */
  public SortableTable getTable() {
    return container;
  }

  /**
   * Construtor de um componente de tabela ordenvel, composta de itens
   * ordenveis.
   * 
   * @param columnFormatter O formatador responsvel pela formatao dos
   *        elementos desse conjunto.
   * @param removeSelectedButton O boto que remove elementos selecionados do
   *        conjunto.
   * @param removeAllButton O boto que remove todos os elementos do conjunto.
   * @param enabled Indica se o container est habilitado ou no.
   */
  ItemContainer(ObjectTableProvider columnFormatter,
    final JButton removeSelectedButton, final JButton removeAllButton,
    final boolean enabled) {
    this(columnFormatter, removeSelectedButton, removeAllButton, enabled, true,
      false);
  }

  /**
   * Construtor de um componente de tabela ordenvel, composta de itens
   * ordenveis.
   * 
   * @param columnFormatter O formatador responsvel pela formatao dos
   *        elementos desse conjunto.
   * @param removeSelectedButton O boto que remove elementos selecionados do
   *        conjunto.
   * @param removeAllButton O boto que remove todos os elementos do conjunto.
   * @param enabled Indica se o container est habilitado ou no.
   * @param enableSort - indica se o container deve ser ordenado
   * @param adjustColumnsWidth - Flag que indica se a tabela deve ser ajustada
   *        para ocupar, pelo menos, o tamanho mnimo para que no seja
   *        necessrio redimensionar suas colunas para que seja possvel ler
   *        todo o contedo destas.
   */
  ItemContainer(ObjectTableProvider columnFormatter,
    final JButton removeSelectedButton, final JButton removeAllButton,
    final boolean enabled, boolean enableSort, boolean adjustColumnsWidth) {
    this.removeSelectedButton = removeSelectedButton;
    this.removeAllButton = removeAllButton;
    this.enabled = enabled;
    this.adjustToMinimumWidth = adjustColumnsWidth;
    itemContainerModel =
      new ObjectTableModel<T>(new ArrayList<T>(), columnFormatter);
    container = new SortableTable(itemContainerModel, enableSort);
    trs = new TableRowSorter<ObjectTableModel>(itemContainerModel);
    container.setRowSorter(trs);
    container.setNoSortStateEnabled(true);
    container.setEnabled(enabled);
    container.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
    /*
     * Se nenhum item estiver selecionado, o boto de ao que opera com um
     * elemento  desativado.
     */
    final ListSelectionModel listModel = container.getSelectionModel();
    listModel.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent ev) {
        if (listModel.getMinSelectionIndex() != -1) {
          removeSelectedButton.setEnabled(enabled);
        }
        else {
          removeSelectedButton.setEnabled(false);
        }
      }
    });
    listModel.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent ev) {
        if (!ev.getValueIsAdjusting()) {
          notifySelectionChange();
        }
      }
    });
    /*
     * Desabilita o boto de remover todos, caso no haja mais nenhum item na
     * lista.
     */
    itemContainerModel.addTableModelListener(new TableModelListener() {
      @Override
      public void tableChanged(TableModelEvent ev) {
        if (itemContainerModel.getRowCount() == 0) {
          removeAllButton.setEnabled(false);
        }
        else {
          removeAllButton.setEnabled(enabled);
        }
      }
    });

    this.view = new JScrollPane(container) {

      /**
       * {@inheritDoc}
       */
      @Override
      public void validate() {
        super.validate();
        if (viewport.isValid() && isPendingColumnAdjustment) {
          adjustLastColumnWidth();
          super.validate();
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              isPendingColumnAdjustment = false;
            }
          });
        }
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public boolean isValid() {
        if (isPendingColumnAdjustment) {
          return false;
        }
        return super.isValid();
      }

    };

    this.view.setPreferredSize(new Dimension(200, 70));
    this.view.validate();
    this.view.addComponentListener(new ComponentListener() {
      @Override
      public void componentHidden(ComponentEvent e) {
      }

      @Override
      public void componentMoved(ComponentEvent e) {
      }

      @Override
      public void componentResized(ComponentEvent e) {
        isPendingColumnAdjustment = true;
        ItemContainer.this.view.validate();
      }

      @Override
      public void componentShown(ComponentEvent e) {
      }
    });

    this.itemListeners = new LinkedList<ItemContainerListener>();
    updateTableAdjustment();
  }

  /**
   * Ajusta o tamanho da ltima coluna para preencher todo o espao deixado
   * entre a tabela e seu viewport.
   */
  private void adjustLastColumnWidth() {
    if (this.container.getColumnCount() == 0) {
      return;
    }
    int viewportWidth = (int) this.view.getViewport().getSize().getWidth();
    int tableWidth = this.container.getWidth();
    if (adjustToMinimumWidth) {
      if (viewportWidth <= minimunSize.width) {

        this.container.setSize(minimunSize);
        doAdjustColumnWidth(minimunSize.width);
        this.container.validate();
        return;
      }

      this.container.setSize(new Dimension(viewportWidth, minimunSize.height));
      this.container.adjustColumnWidth();
      doAdjustColumnWidth(viewportWidth);
      this.container.validate();
      return;
    }
    else {
      if (viewportWidth < tableWidth) {
        return;
      }
    }

    TableColumnModel columnModel = this.container.getColumnModel();
    TableColumn lastColumn =
      columnModel.getColumn(this.container.getColumnCount() - 1);

    lastColumn.setWidth(lastColumn.getPreferredWidth() + viewportWidth
      - tableWidth);
    this.container.validate();

  }

  /**
   * Realiza o ajuste da largura das colunas para que estas ocupem o espao
   * mnimo necessrio para exibir seus contedos. No caso de a tabela estar
   * vazia, todas as colunas tero a mesma largura.
   * 
   * @param width Largura disponvel para a tabela.
   */
  private void doAdjustColumnWidth(int width) {
    int numColumns = this.container.getColumnCount();
    int tableSize = 0;

    if (colunsMinimunSize.isEmpty() || this.container.getRowCount() == 0) {
      int viewportWidth = (int) this.view.getViewport().getSize().getWidth();

      int quotient = viewportWidth / numColumns;
      int modulus = viewportWidth % numColumns;

      for (int i = 0; i < numColumns; i++) {
        int delta = (modulus-- > 0) ? quotient + 1 : quotient;
        TableColumnModel columnModel = this.container.getColumnModel();
        TableColumn column = columnModel.getColumn(i);
        column.setPreferredWidth(delta);
        tableSize += delta;
      }

      return;
    }

    for (int i = 0; i < numColumns; i++) {
      TableColumnModel columnModel = this.container.getColumnModel();
      TableColumn column = columnModel.getColumn(i);
      column.setPreferredWidth(colunsMinimunSize.get(i));
      tableSize += colunsMinimunSize.get(i);
    }

    TableColumnModel columnModel = this.container.getColumnModel();
    TableColumn lastColumn =
      columnModel.getColumn(this.container.getColumnCount() - 1);

    lastColumn.setPreferredWidth(lastColumn.getPreferredWidth() + width
      - tableSize);

    this.container.setSize(new Dimension(width, minimunSize.height));
    return;
  }

  /**
   * Atualiza o tamanho mnimo da tabela.
   */
  private void updateMinimunSize() {
    int width = getMinimunTableWidth();
    int height = getTableHeight();
    minimunSize = new Dimension(width, height);
  }

  /**
   * @return Largura mnima da tabela.
   */
  private int getMinimunTableWidth() {

    int numColumns = this.container.getColumnCount();
    int tableWidth = 0;
    colunsMinimunSize = new ArrayList<Integer>();

    for (int i = 0; i < numColumns; i++) {
      TableColumnModel columnModel = this.container.getColumnModel();
      TableColumn column = columnModel.getColumn(i);
      int width = 0;

      // Obteno da largura do header da coluna
      TableCellRenderer renderer = column.getHeaderRenderer();
      if (renderer == null) {
        renderer = this.container.getTableHeader().getDefaultRenderer();
      }
      Component comp =
        renderer.getTableCellRendererComponent(this.container, column
          .getHeaderValue(), false, false, 0, 0);
      width = comp.getPreferredSize().width;

      // Obteno da largura mnima da coluna
      for (int r = 0; r < this.container.getRowCount(); r++) {
        renderer = this.container.getCellRenderer(r, 0);
        comp =
          renderer.getTableCellRendererComponent(this.container, this.container
            .getValueAt(r, i), false, false, r, i);
        width = Math.max(width, comp.getPreferredSize().width);
      }
      width += this.container.getIntercellSpacing().width * 2;
      colunsMinimunSize.add(i, width);
      tableWidth += width;
    }

    return tableWidth;
  }

  /**
   * @return Altura da tabela.
   */
  private int getTableHeight() {
    int rowCount = this.container.getRowCount();
    int tableHeight = 0;

    for (int i = 0; i < rowCount; i++) {
      int prefRowHeight = 0;
      int colCount = this.container.getColumnCount();
      for (int c = 0; c < colCount; c++) {

        int prefCellHeight = this.container.getRowHeight(i);
        if (prefCellHeight > prefRowHeight) {
          prefRowHeight = prefCellHeight;
        }

      }

      tableHeight += prefRowHeight;
    }

    return tableHeight;

  }

  /**
   * Notifica o evento de mudana de seleo dos elementos.
   */
  private void notifySelectionChange() {
    LinkedList<T> l = new LinkedList<T>();
    int[] indexSelected = this.container.getSelectedRows();
    for (int i = 0; i < indexSelected.length; i++) {
      l.add(itemContainerModel.getRow(indexSelected[i]));
    }
    ItemContainerEvent<T> event = new ItemContainerEvent<T>(this, l);
    Iterator<?> ite = this.itemListeners.iterator();
    while (ite.hasNext()) {
      ItemContainerListener listener = (ItemContainerListener) ite.next();
      listener.wasSelected(event);
    }
  }

  /**
   * Filtra os valores na tabela a partir do que foi digitado no campo do painel
   * de filtro. Usa uma regular expression que pega qualquer parte da string,
   * ignorando maisculas/minsculas
   * 
   * @param toFilter
   */
  public void filter(String toFilter) {
    if (toFilter.isEmpty()) {
      trs.setRowFilter(null);
    }
    else {
      trs.setRowFilter(RowFilter.regexFilter("(?i)" + toFilter));
    }
  }
}
