/*
 * $Id: FooterModelWrapper.java 126580 2012-02-17 19:24:59Z letnog $
 */
package tecgraf.javautils.gui.table;

import java.util.Map;

import javax.swing.event.EventListenerList;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

/**
 * Classe que "decora" ou "engloba" um modelo (<i>Wrapper</i>) com um rodap,
 * isto , exibindo uma informao calculada fixa na ltima linha. A primeira
 * coluna desta ltima linha  reservada para a exibio de um texto, por
 * exemplo "Total".
 * 
 * @author Tecgraf
 */
public abstract class FooterModelWrapper implements TableModel,
  TableModelListener {

  /** Lista de listeners deste modelo */
  private EventListenerList listenerList;
  /** Modelo a ser decorado */
  private TableModel model;
  /** Texto a ser apresentado na coluna 0 (pode ser nulo) */
  private String totalText;
  /** Valores a serem apresentados na ltima linha, por coluna */
  private Map<Integer, Number> footerValues;
  /** Peso da linha. Quanto maior, mais abaixo ser apresentada */
  private int weight;

  /**
   * Cria o decorador de modelo, indicando qual o texto a ser exibido na
   * primeira coluna da ltima linha.
   * 
   * @param model modelo a ser decorado.
   * @param totalText texto a ser exibido na primeira coluna da ltima linha.
   */
  public FooterModelWrapper(TableModel model, String totalText) {
    if (model == null) {
      throw new IllegalArgumentException("model==null");
    }
    weight =
      (FooterModelWrapper.class.isInstance(model)) ? FooterModelWrapper.class
        .cast(model).getWeight() + 1 : 0;
    this.model = model;
    model.addTableModelListener(this);
    this.totalText = totalText;
    this.listenerList = new EventListenerList();
  }

  /**
   * Obtm peso da linha.
   * 
   * @return peso da linha.
   */
  private int getWeight() {
    return weight;
  }

  /**
   * Obtm os valores a serem apresentados na ltima linha.
   * 
   * @return valores a serem apresentados na ltima linha.
   */
  protected abstract Map<Integer, Number> getFooterValues();

  /**
   * Obtm o modelo.
   * 
   * @return modelo.
   */
  public TableModel getModel() {
    return model;
  }

  /**
   * Retorna o nmero de linhas deste modelo, que vai ser igual ao nmero de
   * linhas do modelo decorado + 1 (linha apresentando os totais).
   * 
   * @return 0 se o modelo original estiver vazio, nmero de linhas + 1 caso
   *         contrrio.
   */
  @Override
  public int getRowCount() {
    int originalRows = model.getRowCount();
    return (originalRows == 0) ? 0 : originalRows + 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return model.getColumnCount();
  }

  /**
   * Retorna o valor da clula especificada pelas coordenadas (linha, coluna).
   * Normalmente retorna o valor correspondente da clula no modelo decorado,
   * com exceo da ltima linha, em que a primeira clula exibe um texto e
   * outras exibem um valor calculado.
   * 
   * {@inheritDoc}
   * 
   * @see #FooterModelWrapper(TableModel, String)
   */
  @Override
  public synchronized Object getValueAt(int rowIndex, int columnIndex) {
    if (rowIndex < model.getRowCount()) {
      return model.getValueAt(rowIndex, columnIndex);
    }
    if (columnIndex == 0) {
      return new FooterCell(totalText, weight);
    }
    if (footerValues == null) {
      footerValues = getFooterValues();
    }
    return new FooterCell(footerValues.get(columnIndex), weight);
  }

  /**
   * Se o modelo decorado for editvel, substitui o valor da clula indicada
   * pelas coordenadas (linha, coluna), com exceo da ltima linha (totais),
   * que nunca pode ser editada por ser calculada.
   * 
   * {@inheritDoc}
   */
  @Override
  public synchronized void setValueAt(Object value, int rowIndex,
    int columnIndex) {
    if (rowIndex < model.getRowCount()) {
      model.setValueAt(value, rowIndex, columnIndex);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int columnIndex) {
    return model.getColumnName(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return model.getColumnClass(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    return model.isCellEditable(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addTableModelListener(TableModelListener l) {
    listenerList.add(TableModelListener.class, l);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeTableModelListener(TableModelListener l) {
    listenerList.remove(TableModelListener.class, l);
  }

  /**
   * Notifica os listeners de que este modelo sofreu uma mudana em todas as
   * suas linhas.
   * 
   * @param e informaes sobre o evento ocorrido.
   */
  private void fireTableChanged(TableModelEvent e) {
    Object[] listeners = listenerList.getListenerList(); // No-nulo garantido
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == TableModelListener.class) {
        ((TableModelListener) listeners[i + 1]).tableChanged(e);
      }
    }
  }

  /**
   * Mtodo chamado sempre que ocorre uma alterao no modelo. Refaz os
   * clculos.
   * 
   * @param e informaes sobre o evento.
   */
  @Override
  public void tableChanged(TableModelEvent e) {
    this.footerValues = getFooterValues();
    /* Ao remover a ltima linha, a linha de totais tambm  removida */
    if (e.getType() == TableModelEvent.DELETE && model.getRowCount() == 0) {
      fireTableChanged(new TableModelEvent(this, e.getFirstRow(), e
        .getLastRow() + 1, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
    }
    else {
      fireTableChanged(e);
    }
  }
}
