/*
 * $Id$
 */
package tecgraf.javautils.gui.table;

import java.awt.Component;
import java.awt.FontMetrics;
import java.util.List;

import javax.swing.AbstractListModel;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;

/**
 * Classe que define o header para as linhas de uma tabela.
 */
public class RowHeader extends JList {

  /** Tabela com o header de linha */
  private JTable table;
  /** Ttulos a serem apresentados no header */
  private List<? extends Object> rowNames;
  /** Ttulo do header das linhas */
  private String rowHeaderTitle;
  /** Lista contendo os tooltips de cada linha do rowheader */
  private List<String> rowHeaderTooltips;

  /**
   * Associa um novo cabealho de linha a um painel com uma tabela.
   * 
   * @param pane painel onde o header ser construdo.
   * @param table tabela a ser inserida no painel.
   * @param rowNames lista com os ttulos das linhas.
   * @param rowHeaderTitle ttulo para o header de linha.
   */
  public static void setRowHeader(JScrollPane pane, JTable table,
    List<? extends Object> rowNames, String rowHeaderTitle) {
    pane.setRowHeaderView(new RowHeader(pane, table, rowNames, rowHeaderTitle));
  }

  /**
   * Construtor.
   * 
   * @param pane painel onde o header ser construdo.
   * @param table tabela a ser inserida no painel.
   * @param rowNames lista com os ttulos das linhas.
   * @param rowHeaderTitle ttulo para o header de linha.
   */
  public RowHeader(JScrollPane pane, JTable table,
    List<? extends Object> rowNames, String rowHeaderTitle) {
    this.table = table;
    this.rowNames = rowNames;
    this.rowHeaderTitle = rowHeaderTitle;
    setModel(getRowHeaderModel());
    setFixedCellWidth(getMaxRowNameWidth() + 20);
    setFixedCellHeight(table.getRowHeight());
    setBackground(pane.getBackground());
    setCellRenderer(new RowHeaderRenderer(SwingConstants.LEFT));

    // nova abordagem:
    MultiLineLabelHeaderRenderer renderer = new MultiLineLabelHeaderRenderer();
    renderer.setTitle(table, rowHeaderTitle);
    pane.setCorner(JScrollPane.UPPER_LEFT_CORNER, renderer);
    table.getModel().addTableModelListener(new ModelListener());
  }

  /**
   * Recupera os ttulos do header de linha da tabela.
   * 
   * @return os ttulos do header de linha da tabela.
   */
  public List<? extends Object> getRowNames() {
    return rowNames;
  }

  /**
   * Apresentador da lista que simula o header de linha utilizado nas tabelas de
   * edio de estoque.
   */
  private class RowHeaderRenderer extends JLabel implements ListCellRenderer {
    /**
     * Cria o apresentador utilizando os atributos do header da tabela.
     * 
     * @param horizontalAlignment constante definida em SwingConstants.
     */
    RowHeaderRenderer(int horizontalAlignment) {
      JTableHeader header = table.getTableHeader();
      setOpaque(true);
      setBorder(UIManager.getBorder("TableHeader.cellBorder"));
      setFont(header.getFont());
      setHorizontalAlignment(horizontalAlignment);
      setForeground(header.getForeground());
      setBackground(header.getBackground());
    }

    /**
     * Informa o <code>JLabel</code> a ser apresentado na lista, iniciando o seu
     * valor com o valor da linha da lista.
     * 
     * @param list lista a que pertence a linha.
     * @param value valor atribudo  linha [index]
     * @param index ndice da linha.
     * @param isSelected <code>true</code> se a linha estiver selecionada.
     * @param cellHasFocus <code>true</code> se o foco estiver na linha.
     * 
     * @return o apresentador da linha (<code>JLabel</code>).
     */
    public Component getListCellRendererComponent(JList list, Object value,
      int index, boolean isSelected, boolean cellHasFocus) {
      setText((value == null) ? "" : value.toString());
      if (rowHeaderTooltips != null && index < rowHeaderTooltips.size()) {
        this.setToolTipText(rowHeaderTooltips.get(index));
      }
      return this;
    }
  }

  /**
   * Obtm o modelo da lista do header.
   * 
   * @return modelo da lista do header
   */
  ListModel getRowHeaderModel() {
    return new AbstractListModel() {
      public int getSize() {
        return rowNames.size();
      }

      public Object getElementAt(int index) {
        return rowNames.get(index);
      }
    };
  }

  /**
   * Altera os ttulos do header de linha da tabela.
   * 
   * @param rowNames novos ttulos do header de linha
   */
  public void setRowNames(List<? extends Object> rowNames) {
    this.rowNames = rowNames;
    setFixedCellWidth(getMaxRowNameWidth() + 20);
    setModel(getRowHeaderModel());
  }

  /**
   * Calcula a largura do maior texto do cabealho das linhas.
   * 
   * @return o tamanho mximo do cabealho das linhas
   */
  private int getMaxRowNameWidth() {
    FontMetrics metrics = getFontMetrics(table.getTableHeader().getFont());
    int maxWidth = metrics.stringWidth(rowHeaderTitle);
    for (int i = 0; i < rowNames.size(); i++) {
      String rowName = rowNames.get(i).toString();
      int width = metrics.stringWidth(rowName);
      maxWidth = Math.max(maxWidth, width);
    }
    return maxWidth;
  }

  /**
   * Define os tooltips que devem ser apresentados na row header, para cada
   * linha.
   * 
   * @param tooltips uma lista com os tooltips que devem ser apresentados para
   *        cada linha. Se alguma linha no estiver contida na lista, no ser
   *        exibido nenhum tooltip nela. Se a lista for <code>null</code>,
   *        nenhum tooltip ser exibido.
   */
  public void setRowHeaderTooltips(List<String> tooltips) {
    this.rowHeaderTooltips = tooltips;
  }

  /**
   * Listener de alteraes no modelo da tabela. Ajusta o desenho dos headers
   * quando linhas so adicionadas ou removidas.
   */
  private class ModelListener implements TableModelListener {
    /**
     * {@inheritDoc}
     */
    public void tableChanged(TableModelEvent ev) {
      if (ev.getType() == TableModelEvent.INSERT
        || ev.getType() == TableModelEvent.DELETE) {
        if (ev.getColumn() == TableModelEvent.ALL_COLUMNS) {
          repaint();
        }
      }
    }
  }
}