package tecgraf.javautils.pdfviewer.core;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;

/**
 * Classe responsvel pela leitura (Operaes de IO!) de um {@link InputStream}
 * para um {@link PDFDocument}
 * 
 * @author Tecgraf
 */
public class PDFReader {

  /**
   * Carrega um {@link PDFDocument} de um {@link InputStream}. O
   * {@link InputStream} deve ser fechado pelo caller.
   * 
   * @param inputStream inputStream para ler o {@link PDFDocument}
   * @return documento PDF lido
   * @throws IOException
   */
  public PDFDocument read(InputStream inputStream) throws IOException {
    //carrega o PDF em um array de bytes
    byte[] byteBuf = toByteArray(inputStream);
    ByteBuffer buf = ByteBuffer.wrap(byteBuf);
    return new PDFDocument(buf);
  }

  /**
   * Utilitrio interno de stream para array de bytes.
   * 
   * @param input entrada
   * @return array de bytes
   * @throws IOException em caso de erro de E/S.
   */
  private byte[] toByteArray(InputStream input) throws IOException {
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    copy(input, output);
    return output.toByteArray();
  }

  /**
   * Cpia interna
   * 
   * @param input entrada
   * @param output sada
   * @return tamanho da cpia.
   * @throws IOException em caso de erro de E/S
   */
  private long copyLarge(InputStream input, OutputStream output)
    throws IOException {
    byte[] buffer = new byte[1024 * 1024];
    long count = 0;
    int n = 0;
    while (-1 != (n = input.read(buffer))) {
      output.write(buffer, 0, n);
      count += n;
    }
    return count;
  }

  /**
   * Cpia entre streams
   * 
   * @param input entrada
   * @param output sada
   * @return tamanho da cpia.
   * @throws IOException em caso de erro de E/S
   */
  private int copy(InputStream input, OutputStream output) throws IOException {
    long count = copyLarge(input, output);
    if (count > Integer.MAX_VALUE) {
      return -1;
    }
    return (int) count;
  }

}
