package tecgraf.javautils.pdfviewer.viewer;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.NumberFormat;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.pdfviewer.core.PDFCorePanel;
import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangedListener;

/**
 * Painel que permite ir direto para uma pgina tendo como entrada o nmero
 * 
 * @author Tecgraf
 */
class PageNumberPanel extends JPanel implements PDFPageChangedListener,
  PDFDocumentOpenCloseListener, ActionListener {

  /** Separador do nmero da pgina e do total */
  final static String SEPARATOR = " / ";

  /** Campo de texto que mostra a pgina corrente */
  final JTextField currentPage = new JFormattedTextField(NumberFormat
    .getInstance());

  /** Label com o total de pginas */
  final JLabel totalPagesLabel = new JLabel(SEPARATOR + "0");

  /** Tamanho mximo dos elementos no painel */
  final Dimension elementsMaxSize = new Dimension(50, 20);
  /** Tamanho mnimo dos elementos no painel */
  final Dimension elementsMinSize = new Dimension(20, 10);

  /**
   * Painel bsico associado.
   */
  final PDFCorePanel pdfPanel;

  /**
   * Construtor default
   * 
   * @param pdfPanel painel bsico.
   */
  public PageNumberPanel(final PDFCorePanel pdfPanel) {
    this.pdfPanel = pdfPanel;
    setLayout(new BoxLayout(this, BoxLayout.LINE_AXIS));
    //      setBorder(BorderFactory.createEmptyBorder(0, 5, 0, 5));
    setBorder(BorderFactory.createLineBorder(Color.lightGray));
    setEnabled(false);
    setMaximumSize(new Dimension(elementsMaxSize.width * 2,
      elementsMaxSize.height + 6));
    currentPage.setText("0");
    currentPage.setHorizontalAlignment(JTextField.RIGHT);
    currentPage.setMinimumSize(elementsMinSize);
    currentPage.setPreferredSize(elementsMaxSize);
    currentPage.setMaximumSize(elementsMaxSize);
    totalPagesLabel.setMinimumSize(elementsMinSize);
    totalPagesLabel.setPreferredSize(elementsMaxSize);
    totalPagesLabel.setMaximumSize(elementsMaxSize);
    add(Box.createRigidArea(new Dimension(5, 0)));
    add(currentPage);
    add(totalPagesLabel);
    add(Box.createRigidArea(new Dimension(5, 0)));
    currentPage.addActionListener(this);

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(boolean enabled) {
    super.setEnabled(enabled);
    currentPage.setEnabled(enabled);
    totalPagesLabel.setEnabled(enabled);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void pageChanged(int pageNum, int totalPages) {
    setCurrentPageNumber(pageNum + 1);
  }

  /**
   * Atribui ao campo de texto o nmero da pgina
   * 
   * @param pageNum nmero da pgina
   */
  private void setCurrentPageNumber(int pageNum) {
    currentPage.setText("" + pageNum);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentOpened(PDFCorePanel source) {
    setEnabled(true);
    totalPagesLabel.setText(SEPARATOR + source.getTotalPageNumber());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentClosed(PDFCorePanel source) {
    setEnabled(false);
    //      currentPageModel.setMaximum(new Integer(0));
    currentPage.setText("0");
    totalPagesLabel.setText(SEPARATOR + "0");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    if (e.getSource() == currentPage) {
      try {
        int pageNum = Integer.valueOf(currentPage.getText());
        if (pageNum > 0 && pageNum <= pdfPanel.getTotalPageNumber()) {
          pdfPanel.setPage(pageNum - 1);
        }
        else {
          setCurrentPageNumber(pdfPanel.getPageNumber() + 1);
        }
      }
      catch (NumberFormatException ex) {
        setCurrentPageNumber(pdfPanel.getPageNumber() + 1);
      }
    }
  }
}
