package tecgraf.javautils.pdfviewer.viewer;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.ListCellRenderer;
import javax.swing.plaf.basic.BasicComboBoxEditor;

import tecgraf.javautils.pdfviewer.core.PDFCorePanel;
import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomChangeRequestedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomChangedListener;

/**
 * Combobox que mostra o valor de zoom na tool bar
 * 
 * @author Tecgraf
 */
public class ZoomValueComboBox extends JPanel implements
  PDFZoomChangedListener, PDFDocumentOpenCloseListener {

  /**
   * Altura default do componente
   */
  static private final int DEFAULT_HEIGHT = 26;

  /**
   * Renderiza o texto seguido de um sinal de porcento '%'
   * 
   * @author Tecgraf
   */
  class PercentageListCellRenderer extends DefaultListCellRenderer {
    /**
     * {@inheritDoc}
     */
    @Override
    public Component getListCellRendererComponent(JList list, Object value,
      int index, boolean isSelected, boolean cellHasFocus) {

      JLabel label =
        (JLabel) super.getListCellRendererComponent(list, value, index,
          isSelected, cellHasFocus);

      String zoom = value.toString();
      label.setText(zoom + "%");
      // alinhado no centro para ficar alinhado com o editor da combobox
      label.setHorizontalAlignment(JTextField.CENTER);

      return label;

    }
  }

  /**
   * Editor que coloca um sinal de '%' depois dos nmeros do zoom
   * 
   * @author Tecgraf
   */
  class PercentageComboEditor extends BasicComboBoxEditor {

    /**
     * Painel que segurar o {@link JTextField} e um {@link JLabel} com o '%'
     */
    final JPanel compositeEditor = new JPanel();

    /**
     * Label com o sinal de '%'
     */
    final JLabel pctLabel = new JLabel("%");

    /**
     * Construtor default
     */
    public PercentageComboEditor() {
      compositeEditor.setLayout(new BorderLayout());
      pctLabel.setMinimumSize(new Dimension(25, DEFAULT_HEIGHT));
      // s JLabel opaco tem cor de fundo
      pctLabel.setOpaque(true);
      pctLabel.setBackground(editor.getBackground());
      compositeEditor.add(editor, BorderLayout.CENTER);
      compositeEditor.add(pctLabel, BorderLayout.EAST);
      editor.setHorizontalAlignment(JTextField.RIGHT);
      editor.addKeyListener(new KeyAdapter() {
        // limita os caractres digitados a nmeros, backspace, delete e 4 dgitos mximo
        @Override
        public void keyTyped(KeyEvent e) {
          char c = e.getKeyChar();
          if (getEditor().getText().length() > 3
            || !(Character.isDigit(c) || (c == KeyEvent.VK_BACK_SPACE) || (c == KeyEvent.VK_DELETE))) {
            e.consume();
          }
        }
      });

    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected JTextField createEditorComponent() {
      JTextField editor = new JTextField();
      editor.setBorder(null);
      return editor;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Component getEditorComponent() {
      return compositeEditor;
    }

    /**
     * Retorna {@link JTextField} editor
     * 
     * @return {@link JTextField} editor
     */
    JTextField getEditor() {
      return editor;
    }

    /**
     * Habilita ou desabilita o editor
     * 
     * @param enabled true habilita o editor, false desabilita
     */
    void setEnabled(boolean enabled) {
      editor.setEnabled(enabled);
      pctLabel.setEnabled(enabled);
    }
  }

  /**
   * Notificador que um zoom mudou
   */
  final ItemListener selectionListener = new ItemListener() {

    @Override
    public void itemStateChanged(ItemEvent event) {
      if (event.getStateChange() == ItemEvent.SELECTED) {
        Object item = event.getItem();
        notifyZoomChangeRequested(Double.valueOf(item.toString()));
      }
    }
  };

  /**
   * Valores default da Combo
   */
  final Object values[] = { 400, 300, 250, 200, 150, 125, 100, 75, 50, 25 };

  /**
   * Model com os valores default
   */
  final DefaultComboBoxModel comboModel = new DefaultComboBoxModel(values);

  /** Combox do zoom **/
  final JComboBox comboBox = new JComboBox(comboModel);

  /** Renderer que adiciona o percento no fim do texto do elemento */
  final ListCellRenderer percentCellRenderer = new PercentageListCellRenderer();

  /**
   * Editor com o sinal de % aps o nmero
   */
  final PercentageComboEditor comboEditor = new PercentageComboEditor();

  /**
   * Lista de ouvintes de pedido de mudana de zoom
   */
  final List<PDFZoomChangeRequestedListener> zoomChangeRequestListeners =
    new ArrayList<PDFZoomChangeRequestedListener>();

  /**
   * Construtor default
   */
  public ZoomValueComboBox() {
    super(new BorderLayout());
    setEnabled(false);
    setPreferredSize(new Dimension(75, DEFAULT_HEIGHT));
    setMinimumSize(new Dimension(40, DEFAULT_HEIGHT));
    setMaximumSize(new Dimension(80, DEFAULT_HEIGHT));
    setBorder(BorderFactory.createLineBorder(Color.lightGray));
    comboBox.setMaximumSize(new Dimension(80, DEFAULT_HEIGHT));
    comboBox.setPreferredSize(new Dimension(75, DEFAULT_HEIGHT));
    comboBox.setMinimumSize(new Dimension(40, DEFAULT_HEIGHT));
    add(comboBox, BorderLayout.CENTER);
    comboBox.setEditable(true);
    comboBox.setEditor(comboEditor);
    comboBox.setRenderer(percentCellRenderer);
    comboBox.addItemListener(selectionListener);

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void zoomChanged(double zoomPercentage) {
    comboBox.removeItemListener(selectionListener);
    comboBox.getModel().setSelectedItem((int) zoomPercentage);
    comboBox.addItemListener(selectionListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentOpened(PDFCorePanel pdfViewerPanel) {
    setEnabled(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentClosed(PDFCorePanel source) {
    setEnabled(false);
  }

  /**
   * Adiciona um ouvinte de pedido de mudana de zoom
   * 
   * @param listener ouvinte de pedido de mudana de zoom
   */
  public void addPDFZoomChangeRequestedListener(
    PDFZoomChangeRequestedListener listener) {
    zoomChangeRequestListeners.add(listener);
  }

  /**
   * Remove um ouvinte de pedido de mudana de zoom
   * 
   * @param listener ouvinte de pedido de mudana de zoom
   */
  public void removePDFZoomChangeRequestedListener(
    PDFZoomChangeRequestedListener listener) {
    zoomChangeRequestListeners.remove(listener);
  }

  /**
   * Notifica que houve um pedido de mudana de zoom
   * 
   * @param percentage zoom
   */
  protected void notifyZoomChangeRequested(double percentage) {
    for (PDFZoomChangeRequestedListener listener : zoomChangeRequestListeners) {
      listener.zoomChangeRequested(percentage);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(boolean enabled) {
    comboBox.setEnabled(enabled);
    comboEditor.setEnabled(enabled);
  }
}
