package tecgraf.javautils.pdfviewer.core;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangeRequestedListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFPageChangedListener;

import com.sun.pdfview.PDFFile;
import com.sun.pdfview.PDFPage;
import com.sun.pdfview.PageChangeListener;

/**
 * Painel de ThumbNails do painel PDF.
 *
 * @author Tecgraf
 */
public class PDFCoreThumbnailPanel extends JPanel implements
  PDFDocumentOpenCloseListener, PDFPageChangedListener {

  /**
   * Table model que fornece uma PDFPage por linha da tabela
   *
   * @author Tecgraf
   */
  class PDFFileTableModel extends DefaultTableModel {

    /** Arquivo PDF a ser exibido */
    private PDFFile pdfFile;

    /**
     * Construtor default
     */
    public PDFFileTableModel() {
      super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      if (pdfFile == null) {
        return super.getRowCount();
      }
      return pdfFile.getNumPages();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      // apenas 1 coluna para mostrar a pgina do documento PDF
      return 1;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int row, int column) {
      return pdfFile.getPage(row + 1);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(int row, int column) {
      return false;
    }

    /**
     * setter do pdfFile, pode ser nulo para dizer que no h arquivo pdf
     *
     * @param pdfFile arquivo pdf
     */
    void setPDFFile(PDFFile pdfFile) {
      this.pdfFile = pdfFile;
      fireTableDataChanged();
    }

  }

  /** ScrollPane que segura as imagens das pginas */
  final JScrollPane scrollPane = new JScrollPane();

  /** Tamanho da thumbnail, a referncia  imutvel, mas o width e height no */
  final Dimension thumbnailSize = new Dimension(0, 0);

  /** Aspect ratio da pgina (height dividido por width) */
  double pageAspectRatio = -1;

  /** Listener que  notificado da mudana de pgina do painel PDF */
  final PageChangeListener thumbPageChangeListener = new PageChangeListener() {

    @Override
    public void gotoPage(int pageNumber) {
      notifyPageChangeRequestedListeners(pageNumber + 1);
    }
  };

  /** Lista de listeners que so notificados da inteno de mudana de pgina */
  private List<PDFPageChangeRequestedListener> pageChangeRequestedListeners =
          new ArrayList<>();

  /** Tabela qual cada linha  uma pgina do documento PDF */
  final private JTable table = new JTable();

  /** Model para exibir os thumbnails */
  final private PDFFileTableModel pdfFileTableModel = new PDFFileTableModel();

  /**
   * Construtor default
   */
  public PDFCoreThumbnailPanel() {
    super(new BorderLayout());
    setMinimumSize(new Dimension(150, 200));
    setupTable();
    add(scrollPane, BorderLayout.CENTER);
    scrollPane.setViewportView(table);
    addComponentListener(new ComponentAdapter() {
      @Override
      public void componentResized(ComponentEvent e) {
        calculateThumbnailSizeFromPanel();
      }
    });
  }

  /**
   * Calcula o tamanho do thumbnail de acordo com o width deste painel
   */
  protected void calculateThumbnailSizeFromPanel() {
    if (pageAspectRatio > 0) {
      int panelWidth = getWidth();
      int thumbWidth = (panelWidth - 40);
      if (thumbWidth < 10) {
        thumbWidth = 10;
      }
      int thumbHeight = (int) (thumbWidth * pageAspectRatio);

      thumbnailSize.width = thumbWidth;
      thumbnailSize.height = thumbHeight;
      table.setRowHeight(thumbHeight + 50);
    }
  }

  /**
   * Configura a tabela (atribui o Model, Renderer & Listener)
   */
  private void setupTable() {
    table.setAutoCreateColumnsFromModel(false);
    TableColumn singleColumn = new TableColumn(0);
    table.getColumnModel().addColumn(singleColumn);
    TableCellRenderer renderer = new ThumbnailTableCellRenderer(thumbnailSize);
    table.getColumnModel().getColumn(0).setCellRenderer(renderer);
    table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    table.setGridColor(Color.LIGHT_GRAY); // desabilita as bordas das clulas
    table.setModel(pdfFileTableModel);

    // adiciona o listener que faz o pedido de mudana de pgina
    table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent event) {
          if (event.getValueIsAdjusting()) {
            return;
          }
          int index = table.getSelectedRow();

          if (index > -1) {
            notifyPageChangeRequestedListeners(index);
          }
        }
      });

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentOpened(PDFCorePanel pdfViewerPanel) {
    PDFFile pdfFile = pdfViewerPanel.getPDFFile();
    calculatePageAspectRatio(pdfFile);
    pdfFileTableModel.setPDFFile(pdfFile);
  }

  /**
   * Calcula o aspect ratio do papel do documento PDF passado por argumento. A
   * pgina usada no clculo  a pgina de nmero 1 (primeira pgina)
   *
   * @param pdfFile documento PDF a ter o aspect ratio lido
   */
  private void calculatePageAspectRatio(PDFFile pdfFile) {
    PDFPage firstPage = pdfFile.getPage(1);
    // as contantes 100 abaixo so apenas para pegar o aspect ratio original
    Dimension strechedPageDimension = new Dimension(100, 100);
    Dimension pageDimension =
      firstPage.getUnstretchedSize(strechedPageDimension.width,
        strechedPageDimension.height, null);
    pageAspectRatio = (double) pageDimension.height / pageDimension.width;
    calculateThumbnailSizeFromPanel();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void documentClosed(PDFCorePanel source) {
    pageAspectRatio = -1;
    thumbnailSize.width = 0;
    thumbnailSize.height = 0;
    pdfFileTableModel.setPDFFile(null);

  }

  /**
   * Notifica os {@link PDFPageChangedListener} de uma inteno de mudana de
   * pgina
   *
   * @param pageNumber nmero da pgina
   */
  protected void notifyPageChangeRequestedListeners(int pageNumber) {
    for (PDFPageChangeRequestedListener listener : pageChangeRequestedListeners) {
      listener.onPageChangeRequested(pageNumber);
    }
  }

  /**
   * Adiciona um listener de pedido de mudana de pgina
   *
   * @param listener listener de pedido de mudana de pgina
   */
  public void addPDFPageChangeRequestListener(
    PDFPageChangeRequestedListener listener) {
    pageChangeRequestedListeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void pageChanged(int pageNum, int totalPages) {
    if (table.getSelectedRow() != pageNum) {
      table.getSelectionModel().setSelectionInterval(pageNum, pageNum);
      table.scrollRectToVisible(table.getCellRect(pageNum, 0, true));
    }

  }
}
