package tecgraf.javautils.sparkserver.utils;

import java.util.HashSet;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.annotations.Api;
import io.swagger.jaxrs.Reader;
import io.swagger.jaxrs.config.BeanConfig;
import io.swagger.models.Swagger;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.core.JuIController;

public class JuSwaggerParser {

  private final static Logger logger = LoggerFactory.getLogger(JuSwaggerParser.class);

  public static String getSwaggerJson(String title, String packageName) throws JsonProcessingException {
    Swagger swagger = getSwagger(title, packageName);
    String json = swaggerToJson(swagger);
    return json;
  }

  public static String getSwaggerJson(JuIController controller) throws JsonProcessingException {
    final String title = controller.getClass().getSimpleName().toLowerCase();
    Swagger swagger = getSwagger(title, controller.getClass().getPackageName());
    String json = swaggerToJson(swagger);
    return json;
  }

  public static Swagger getSwagger(String title, String packageName) {
    Reflections reflections = new Reflections(packageName);
    BeanConfig beanConfig = new BeanConfig();
    beanConfig.setTitle(title);
    beanConfig.setResourcePackage(packageName);
    beanConfig.setScan(true);
    beanConfig.scanAndRead();
    Swagger swagger = beanConfig.getSwagger();

    Reader reader = new Reader(swagger);

    Set<Class<?>> apiClasses = reflections.getTypesAnnotatedWith(Api.class);
    for (Class<?> apiClass : apiClasses) {
      logger.info("API class: " + apiClass.getName());
    }
    return reader.read(apiClasses);
  }

  public static Swagger getSwagger(String title, Class<? extends JuIController> controllerClass) {
    logger.info("Search swagger for class: " + controllerClass);

    final Set<Class<?>> loadClasses = new HashSet<Class<?>>();
    Class<?> cls = controllerClass;
    while (cls != null) {
      loadClasses.add(cls);
      cls = cls.getSuperclass();
    }
    BeanConfig beanConfig = new BeanConfig();
    beanConfig.setTitle(title);
    beanConfig.setScan(true);
    beanConfig.scanAndRead();

    Swagger swagger = beanConfig.getSwagger();
    final Reader reader = new Reader(swagger);
    for (Class<?> loadClass : loadClasses) {
      logger.info("Swagger API class: " + loadClass.getName());
    }
    return reader.read(loadClasses);
  }

  public static String swaggerToJson(Swagger swagger) throws JsonProcessingException {
    ObjectMapper objectMapper = new ObjectMapper();
    objectMapper.setSerializationInclusion(JsonInclude.Include.NON_EMPTY);
    String json = objectMapper.writeValueAsString(swagger);
    return json;
  }

}

