package tecgraf.javautils.sparkserver.standard;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.JsonProcessingException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.core.JuIWebSocket;
import tecgraf.javautils.sparkserver.utils.JuResponseUtilities;
import tecgraf.javautils.sparkserver.utils.JuStringUtilities;
import tecgraf.javautils.sparkserver.utils.JuSwaggerParser;
import tecgraf.javautils.sparkserver.core.JuIController;
import tecgraf.javautils.sparkserver.core.JuIEndpoint;

import spark.Route;
import spark.Service;

public class JuServer {

  private static JuServer instance;

  private final Service server;

  private final List<JuIController> controllers = new ArrayList<>();

  private final Logger logger = LoggerFactory.getLogger(JuServer.class);

  private String contactName;
  private String name;
  private String version;
  private String description;
  private String hostAddress;

  private JuServer() {
    this.server = Service.ignite();
    setPort(9999);
    setThreadsPool(5, 3, 3000);
  }

  static public JuServer getInstance() {
    if (instance == null) {
      instance = new JuServer();
    }
    return instance;
  }

  public void setThreadsPool(int maxThreads, int minThreads, int timeOutMillis) {
    this.server.threadPool(maxThreads, minThreads, timeOutMillis);
  }

  public void setPort(int port) {
    this.server.port(port);
  }

  public int getHostPort() {
    return this.server.port();
  }

  public void setExceptionHandler(Consumer<Exception> consumer) {
    this.server.initExceptionHandler(consumer);
  }

  public void stop() {
    this.server.stop();
  }

  public void start() throws Exception {
    try {
      controllers.forEach(c -> registerControllerWebSockets(c));
      controllers.forEach(c -> registerControllerEndpoints(c));
      registerSwagger();
      this.enableCORS();
    }
    catch (Throwable t) {
      this.server.stop();
      throw new Exception(t);
    }
    this.server.init();
    this.server.awaitInitialization();
  }

  public Stream<JuIController> getControllers() {
    return controllers.stream();
  }

  public void addController(JuIController controller) {
    if (controller == null) {
      throw new IllegalArgumentException("null controller not allowed!");
    }
    this.controllers.add(controller);
  }

  /**
   * Habilita liberação total ('*') do CORS.
   */
  private void enableCORS() {
    enableCORS("*", "*", "*");
  }

  /**
   * Habilita CORS com base nos parâmetros da requisição.
   * @param origin origem
   * @param methods métodos
   * @param headers cabeçalhos
   */
  private void enableCORS(final String origin, final String methods, final String headers) {
    this.server.options("/*", (request, response) -> {

      String accessControlRequestHeaders = request.headers("Access-Control-Request-Headers");
      if (accessControlRequestHeaders != null) {
        response.header("Access-Control-Allow-Headers", accessControlRequestHeaders);
      }

      String accessControlRequestMethod = request.headers("Access-Control-Request-Method");
      if (accessControlRequestMethod != null) {
        response.header("Access-Control-Allow-Methods", accessControlRequestMethod);
      }

      return "OK";
    });

    this.server.before((request, response) -> {
      response.header("Access-Control-Allow-Origin", origin);
      response.header("Access-Control-Request-Method", methods);
      response.header("Access-Control-Allow-Headers", headers);
      // Note: this may or may not be necessary in your particular application
      response.type("application/json");
    });
  }

  private void registerSwagger() throws Exception {
    try {
      final String swaggerJson = JuSwaggerParser.getSwaggerJsonString(this);
      final String path = "/swagger/";
      this.server.get(path, (req, res) -> JuResponseUtilities.setResponseAsJsonObject(res, 200, swaggerJson));
      logger.info("Swagger for ...");
    }
    catch (JsonProcessingException e) {
      throw new Exception(e);
    }
  }

  private void registerControllerEndpoints(JuIController controller) {
    final Stream<JuIEndpoint> endpoints = controller.getEndpoints();
    endpoints.forEach(ed -> registerEndpoint(controller, ed));
  }

  private void registerControllerWebSockets(JuIController controller) {
    final Stream<JuIWebSocket> websockets = controller.getWebSockets();
    websockets.forEach(ws -> registerWebSocket(controller, ws));
  }

  private void registerEndpoint(JuIController controller, JuIEndpoint endpoint) {
    final Route route = endpoint.getRoute();
    final JuVerb verb = endpoint.getVerb();
    final String path = JuStringUtilities.getRealPath(controller, endpoint.getPath());

    final String fmt = "Register endpoint '%s' for controller '%s' at path '%s'";
    logger.info(String.format(fmt, endpoint.getName(), controller.getName(), path));
    switch (verb) {
      case GET:
        this.server.get(path, route);
        return;
      case PUT:
        this.server.put(path, route);
        return;
      case POST:
        this.server.post(path, route);
        return;
      case DELETE:
        this.server.delete(path, route);
        return;
      default:
        throw new IllegalArgumentException("Verb not recognized!");
    }
  }

  private void registerWebSocket(JuIController controller, JuIWebSocket websocket) {
    final Class<? extends JuWebSocketClass> clazz = websocket.getWebSocketClass();
    final String path = "/" + JuStringUtilities.getRealPath(controller, websocket.getPath());
    final String fmt = "Register websocket '%s' for controller '%s' at path '%s'";
    logger.info(String.format(fmt, websocket.getWebSocketClass().getName(), controller.getName(), path));
    this.server.webSocket(path, clazz);
  }

  public String getContactName() {
    return contactName;
  }

  public JuServer setContactName(String contactName) {
    this.contactName = contactName;
    return this;
  }

  public String getDescription() {
    return description == null ? "" : description;
  }

  public JuServer setDescription(String description) {
    this.description = description;
    return this;
  }

  public String getName() {
    return name == null ? this.getClass().getSimpleName() : name;
  }

  public JuServer setName(String name) {
    this.name = name;
    return this;
  }

  public String getVersion() {
    return version == null ? "V???.???.???" : version;
  }

  public JuServer setVersion(String version) {
    this.version = version;
    return this;
  }

  public String getHostAddress() {
    return hostAddress == null ? "localhost" : hostAddress;
  }

  public JuServer setHostAddress(String hostAddress) {
    this.hostAddress = hostAddress;
    return this;
  }

}
