package tecgraf.javautils.sparkserver.utils;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import javax.ws.rs.core.MediaType;

import io.swagger.models.Contact;
import io.swagger.models.Info;
import io.swagger.models.Operation;
import io.swagger.models.Path;
import io.swagger.models.RefModel;
import io.swagger.models.Response;
import io.swagger.models.Scheme;
import io.swagger.models.Swagger;
import io.swagger.models.Tag;
import io.swagger.models.parameters.PathParameter;
import io.swagger.models.parameters.QueryParameter;

import tecgraf.javautils.sparkserver.core.JuIController;
import tecgraf.javautils.sparkserver.core.JuIEndpoint;
import tecgraf.javautils.sparkserver.core.JuIPathParameter;
import tecgraf.javautils.sparkserver.core.JuIQueryParameter;
import tecgraf.javautils.sparkserver.standard.JuServer;

public class JuSwaggerReader {

  final private JuServer server;
  final private Swagger swagger;

  public JuSwaggerReader(JuServer server) {
    this.server = server;
    this.swagger = new Swagger();
  }

  public Swagger read() {
    final Info info = findInfo();
    swagger.setInfo(info);
    swagger.setHost(server.getHostAddress() + ":" + server.getHostPort());
    swagger.setSchemes(Collections.singletonList(Scheme.HTTP));
    this.server.getControllers().forEach(ctrl -> {
      read(ctrl);
    });
    return swagger;
  }

  private void read(JuIController controller) {
    controller.getEndpoints().forEach(ep -> {
      final Tag tag = new Tag();
      tag.name(controller.getName());
      tag.description(controller.getDescription());
      swagger.tag(tag);
      read(controller, ep);
    });
  }

  private void read(JuIController controller, JuIEndpoint endpoint) {
    final Path path = new Path();
    final Operation op = new Operation();
    op.description(endpoint.getDescription());
    op.setTags(Collections.singletonList(controller.getName()));

    final List<String> consumes = findResponseConsume(endpoint);
    op.consumes(consumes);

    final List<String> produces = findResponseProduce(endpoint);
    op.produces(produces);

    final Set<JuIPathParameter> pps = endpoint.getPathParameters();
    pps.forEach( pp -> {
      final PathParameter parameter = new PathParameter();
      parameter.setName(pp.getName());
      parameter.setDescription(pp.getDescription());
      parameter.setExample(pp.getExampleAsString());
      parameter.setType(pp.getClassValue().getSimpleName().toLowerCase());
      op.addParameter(parameter);
    });

    final Set<JuIQueryParameter> qps = endpoint.getQueryParameters();
    qps.forEach( qp -> {
      final QueryParameter parameter = new QueryParameter();
      parameter.setName(qp.getName());
      parameter.setDescription(qp.getDescription());
      parameter.setExample(qp.getExampleAsString());
      parameter.setType(qp.getClassValue().getSimpleName().toLowerCase());
      op.addParameter(parameter);
    });

    final Response response = new Response();
    response.description("ok");
    final RefModel model = new RefModel();
    model.set$ref("abc");
    model.setDescription("yyy");
    response.setResponseSchema(model);
    op.response(200, response);
    op.setSchemes(Collections.singletonList(Scheme.HTTP));

    path.set(findOperationName(endpoint), op);
    final String fullPath = findFullPath(controller, endpoint);
    swagger.path(fullPath, path);
  }

  private List<String> findResponseProduce(JuIEndpoint endpoint) {
    final Class responseClass = endpoint.getRoute().getResponseClass();
    final List<String> consumes = responseClass.equals(String.class) ?
      Collections.singletonList(MediaType.TEXT_PLAIN) :
      Collections.singletonList(MediaType.APPLICATION_JSON);
    return consumes;
  }

  private List<String> findResponseConsume(JuIEndpoint endpoint) {
    final Class responseClass = endpoint.getRoute().getResponseClass();
    final List<String> consumes = responseClass.equals(String.class) ?
      Collections.singletonList(MediaType.TEXT_PLAIN) :
      Collections.singletonList(MediaType.APPLICATION_JSON);
    return consumes;
  }

  private Info findInfo() {
    final Contact contact = new Contact();
    contact.setName(server.getContactName());

    final Info info = new Info();
    info.contact(contact);
    info.description(server.getDescription());
    info.setTitle(server.getName());
    info.setVersion(server.getVersion());
    return info;
  }

  private String findFullPath(JuIController controller, JuIEndpoint endpoint) {
    String pth = "/" + JuStringUtilities.getRealPath(controller, endpoint.getPath());
    pth = pth.replaceAll(":(\\w+)", "{$1}");
    return pth;
  }

  private String findOperationName(JuIEndpoint endpoint) {
    return endpoint.getVerb().name().toLowerCase();
  }


}
