package tecgraf.javautils.sparkserver.swagger;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.models.Swagger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JuSwaggerWriter {

  private static final Logger logger = LoggerFactory.getLogger(JuSwaggerWriter.class);

  public String prepareUi(Swagger swagger, final JuSwaggerTheme theme) throws IOException {
    final String baseDirectory = getBaseDirectory();
    final String outputDirectory = getOutputDirectory(baseDirectory);
    logger.info("swagger ui local directory: " + outputDirectory);
    logger.info("swagger ui base directory: " + baseDirectory);
    deleteDirectory(outputDirectory);
    createDirectory(outputDirectory);
    unzipFilesTo(outputDirectory);
    generateSpecFile(outputDirectory, swagger);
    applyTheme(outputDirectory, theme);
    return baseDirectory;
  }

  private void generateSpecFile(String uiFolder, Swagger swagger) throws IOException {
    final String fileName = "swagger-spec.js";
    final String pathName = uiFolder + "/" + fileName;
    final ObjectMapper mapper = new ObjectMapper(new JsonFactory());
    mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    final String jsonText = mapper.writeValueAsString(swagger);
    final String jsText = "window.swaggerSpec=" + jsonText;
    File file = new File(pathName);
    file.delete();
    try (PrintWriter out = new PrintWriter(pathName)) {
      out.println(jsText);
    }
    logger.info("swagger definitions saved to '" + pathName + "");
  }

  private void unzipFilesTo(String uiFolder) throws IOException {
    // All files are in zip root directory!
    final InputStream is = getClass().getResourceAsStream("/swagger-ui.zip");
    final ZipInputStream zis = new ZipInputStream(is);
    final byte[] buffer = new byte[2048];
    ZipEntry entry;
    Path outDirPath = Paths.get(uiFolder);
    while ((entry = zis.getNextEntry()) != null) {
      final String name = entry.getName();
      final Path outPath = outDirPath.resolve(name);
      final File outFile = outPath.toFile();
      try (FileOutputStream fos = new FileOutputStream(outFile);
        BufferedOutputStream bos = new BufferedOutputStream(fos, buffer.length)) {
        int len;
        while ((len = zis.read(buffer)) > 0) {
          bos.write(buffer, 0, len);
        }
      }
    }
  }

  private void applyTheme(String uiFolder, final JuSwaggerTheme theme) throws IOException {
    final String themeCss = readFile(uiFolder, theme.getValue() + ".css", StandardCharsets.UTF_8);
    saveFile(uiFolder, "swagger-ui.css", themeCss);
    logger.info("css theme successfully applied: " + theme);
  }

  private String readFile(String uiFolder, String name, Charset encoding) throws IOException {
    byte[] encoded = Files.readAllBytes(Paths.get(uiFolder, name));
    return new String(encoded, encoding);
  }

  private void saveFile(String uiFolder, String fileName, String content) throws IOException {
    File file = new File(uiFolder, fileName);
    file.delete();

    FileWriter f2 = new FileWriter(file, false);
    f2.write(content);
    f2.close();
    logger.debug("Spark-Swagger: Swagger UI file " + fileName + " successfully saved");
  }

  public String getOutputDirectory(String baseDirectory) {
    return baseDirectory + "/swagger-ui/";
  }

  private String getBaseDirectory() {
    //    final String tmpDir = System.getProperty("java.io.tmpdir");
    //    final String tmpDir = "/home/macplinio/tmp/";
    final String name = UUID.randomUUID().toString();
    final String tmpDir = "/tmp/" + name + "/";
    return tmpDir;
  }
  public void createDirectory(final String pathName) throws IOException {
    final Path path = Paths.get(pathName);
    Files.createDirectories(path);
  }

  private boolean deleteDirectory(String pathName) {
    File directoryToBeDeleted = new File(pathName);
    File[] allContents = directoryToBeDeleted.listFiles();
    if (allContents != null) {
      for (File file : allContents) {
        deleteDirectory(file.getPath());
      }
    }
    return directoryToBeDeleted.delete();
  }
}
