package tecgraf.javautils.sparkserver.library.standard;

import java.util.List;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import tecgraf.javautils.sparkserver.library.core.JuIFunction;
import tecgraf.javautils.sparkserver.library.exceptions.JuException;
import tecgraf.javautils.sparkserver.library.utils.JuResponseUtilities;
import tecgraf.javautils.sparkserver.library.utils.JuSparkUtilities;

import spark.Request;
import spark.Response;
import spark.Route;

public class JuRoute<T> implements Route {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());
  private final JuIFunction<T> function;
  private final Class<T> responseClass;
  private final Class<?> containedClass;
  private boolean container;

  public JuRoute(JuIFunction<T> function, Class<T> responseClass) {
    this(function, responseClass, null);
  }

  public JuRoute(JuIFunction<T> function, Class<T> responseClass, Class<?> containedClass) {
    if (function == null) {
      throw new IllegalArgumentException("null function not allowed!");
    }
    this.function = function;
    this.responseClass = responseClass;
    this.container = List.class.isAssignableFrom(responseClass);
    this.containedClass = containedClass;
  }

  final public Class<T> getResponseClass() {
    return responseClass;
  }

  public Class<?> getContainedClass() {
    return containedClass;
  }

  public final boolean isContainer() {
    return container;
  }

  @Override
  final public Object handle(Request request, Response response) throws Exception {
    final String hashString = UUID.randomUUID().toString();
    try {
      logger.info(getLoggerId(">>", null, hashString, request.pathInfo()));
      final T object = function.call(request, response);
      if (!responseClass.isInstance(object)) {
        final String fmt = "route returned a bad object (expected class was %s, but found %s)";
        throw new Exception(String.format(fmt, responseClass.getName(), object.getClass().getName()));
      }
      logger.info(getLoggerId("<<", 200, hashString, String.format("status %d", response.status())));
      return JuSparkUtilities.objectToJson(object);
    }
    catch (JuException e) {
      final String err = e.getMessage();
      final int statusCode = e.getStatusCode();
      logger.warn(getLoggerId("XX", statusCode, hashString, err));
      return JuResponseUtilities.setResponseAsText(response, statusCode, err);
    }
    catch (Throwable t) {
      final String err = t.getMessage();
      final int statusCode = 500;
      logger.error(getLoggerId("!!", statusCode, hashString, err), t);
      return JuResponseUtilities.setResponseAsText(response, statusCode, err);
    }
  }

  private String getLoggerId(String prefix, Integer statusCode, String hashString, String text) {
    if (statusCode == null) {
      return String.format("%s [%s]: %s", prefix, hashString, text);
    }
    return String.format("%s [%s] (%02d): %s", prefix, hashString, statusCode, text);
  }

}