package tecgraf.javautils.sparkserver.library.swagger;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.ws.rs.core.MediaType;

import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.models.ArrayModel;
import io.swagger.models.Contact;
import io.swagger.models.Info;
import io.swagger.models.Model;
import io.swagger.models.ModelImpl;
import io.swagger.models.Operation;
import io.swagger.models.Path;
import io.swagger.models.Response;
import io.swagger.models.Scheme;
import io.swagger.models.Swagger;
import io.swagger.models.Tag;
import io.swagger.models.parameters.PathParameter;
import io.swagger.models.parameters.QueryParameter;
import io.swagger.models.properties.ArrayProperty;
import io.swagger.models.properties.BooleanProperty;
import io.swagger.models.properties.DoubleProperty;
import io.swagger.models.properties.FloatProperty;
import io.swagger.models.properties.IntegerProperty;
import io.swagger.models.properties.LongProperty;
import io.swagger.models.properties.ObjectProperty;
import io.swagger.models.properties.Property;
import io.swagger.models.properties.StringProperty;

import tecgraf.javautils.sparkserver.library.core.JuIController;
import tecgraf.javautils.sparkserver.library.core.JuIEndpoint;
import tecgraf.javautils.sparkserver.library.core.JuIPathParameter;
import tecgraf.javautils.sparkserver.library.core.JuIQueryParameter;
import tecgraf.javautils.sparkserver.library.core.JuIResponse;
import tecgraf.javautils.sparkserver.library.standard.JuServer;
import tecgraf.javautils.sparkserver.library.utils.JuStringUtilities;

import static io.swagger.util.PrimitiveType.createProperty;

public class JuSwaggerReader {

  final private int MAX_LEVEL = 10;

  final private JuServer server;
  final private Swagger swagger;

  public JuSwaggerReader(JuServer server) {
    this.server = server;
    this.swagger = new Swagger();
  }

  public Swagger read() {
    final Info info = findInfo();
    swagger.setInfo(info);
    swagger.setHost(server.getHostAddress() + ":" + server.getHostPort());
    swagger.setSchemes(Collections.singletonList(Scheme.HTTP));
    this.server.getControllers().forEach(ctrl -> {
      readController(ctrl);
    });
    return swagger;
  }

  private void readController(JuIController controller) {
    controller.getEndpoints().forEach(ep -> {
      final Tag tag = new Tag();
      tag.name(controller.getName());
      tag.description(controller.getDescription());
      swagger.tag(tag);
      readEndpoint(controller, ep);
    });
  }

  private void readEndpoint(JuIController controller, JuIEndpoint endpoint) {
    final Path path = new Path();
    final Operation op = new Operation();
    op.description(endpoint.getDescription());
    op.setTags(Collections.singletonList(controller.getName()));

    final List<String> consumes = findResponseConsume(endpoint);
    op.consumes(consumes);

    final List<String> produces = findResponseProduce(endpoint);
    op.produces(produces);

    insertParameters(endpoint, op);
    insertResponses(endpoint, op);
    op.setSchemes(Collections.singletonList(Scheme.HTTP));

    path.set(findOperationName(endpoint), op);
    final String fullPath = findFullPath(controller, endpoint);
    swagger.path(fullPath, path);
  }

  private void insertResponses(JuIEndpoint endpoint, Operation op) {
    final Collection<JuIResponse> rps = endpoint.getResponses();
    rps.forEach(rp -> {
      final Response response = new Response();
      response.setDescription(rp.getDescription());
      final Map<String, Object> examples = new HashMap<>();
      final HashMap<String, Object> exs = rp.getExamplesAsString();
      exs.forEach((name, value) -> {
        examples.put(name, value);
      });
      response.setExamples(examples);
      final int statusCode = rp.getStatusCode();
      final Class responseClass = endpoint.getRoute().getResponseClass();
      final Class containeredClass = endpoint.getRoute().getContainedClass();
      if (statusCode == 200 && !isClassPrimitive(responseClass)) {
        final Model model = createModel(responseClass, containeredClass);
        response.setResponseSchema(model);
      }
      op.addResponse("" + statusCode, response);
    });
  }

  private void insertParameters(JuIEndpoint endpoint, Operation op) {
    final Set<JuIPathParameter> pps = endpoint.getPathParameters();
    pps.forEach(pp -> {
      final PathParameter parameter = new PathParameter();
      parameter.setName(pp.getName());
      parameter.setDescription(pp.getDescription());
      parameter.setExample(pp.getExampleAsString());
      parameter.setType(pp.getClassValue().getSimpleName().toLowerCase());
      op.addParameter(parameter);
    });

    final Set<JuIQueryParameter> qps = endpoint.getQueryParameters();
    qps.forEach(qp -> {
      final QueryParameter parameter = new QueryParameter();
      parameter.setName(qp.getName());
      parameter.setDescription(qp.getDescription());
      parameter.setExample(qp.getExampleAsString());
      parameter.setType(qp.getClassValue().getSimpleName().toLowerCase());
      op.addParameter(parameter);
    });
  }

  private List<String> findResponseProduce(JuIEndpoint endpoint) {
    final Class responseClass = endpoint.getRoute().getResponseClass();
    final List<String> consumes = isClassPrimitive(responseClass) ?
      Collections.singletonList(MediaType.TEXT_PLAIN) :
      Collections.singletonList(MediaType.APPLICATION_JSON);
    return consumes;
  }

  private List<String> findResponseConsume(JuIEndpoint endpoint) {
    final Class responseClass = endpoint.getRoute().getResponseClass();
    final List<String> consumes = isClassPrimitive(responseClass) ?
      Collections.singletonList(MediaType.TEXT_PLAIN) :
      Collections.singletonList(MediaType.APPLICATION_JSON);
    return consumes;
  }

  private Info findInfo() {
    final Contact contact = new Contact();
    contact.setName(server.getContactName());

    final Info info = new Info();
    info.contact(contact);
    info.description(server.getDescription());
    info.setTitle(server.getName());
    info.setVersion(server.getVersion());
    return info;
  }

  private String findFullPath(JuIController controller, JuIEndpoint endpoint) {
    String pth = "/" + JuStringUtilities.getRealPath(controller, endpoint.getPath());
    pth = pth.replaceAll(":(\\w+)", "{$1}");
    return pth;
  }

  private String findOperationName(JuIEndpoint endpoint) {
    return endpoint.getVerb().name().toLowerCase();
  }

  private boolean isClassPrimitive(Class<?> clazz) {
    Class<?>[] list =
      new Class<?>[] { String.class, Integer.class, Long.class, Boolean.class, Short.class, Double.class, Float.class,
        int.class, double.class, float.class, char.class, long.class, short.class, boolean.class };
    return !Arrays.stream(list).filter(c -> c.equals(clazz)).findFirst().isEmpty();
  }

  private Model createModel(Class clazz, Class containedClass) {
    if (clazz.isEnum()) {
      return createEnumModel(clazz);
    }
    else if ((List.class.isAssignableFrom(clazz) || clazz.isArray())) {
      return createArrayModel(containedClass);
    }
    return createClassModel(clazz);
  }

  private ArrayModel createArrayModel(Class containedClass) {
    final ArrayModel model = new ArrayModel();
    if (containedClass != null) {
      final String name = "elem";
      final Property property = createClassProperty(1, name, containedClass);
      model.setItems(property);
    }
    return model;
  }

  private ModelImpl createClassModel(Class clazz) {
    final ModelImpl model = new ModelImpl();
    model.setType(ModelImpl.OBJECT);
    final List<Field> fields = getAnnotatedFields(clazz);
    for (Field field : fields) {
      model.addProperty(getFieldName(field), createFieldProperty(1, field));
    }
    return model;
  }

  private Property createClassProperty(int level, String name, Class clazz) {
    if (level >= MAX_LEVEL) {
      final ObjectProperty property = new ObjectProperty();
      property.setName(name);
      return property;
    }
    if (isClassPrimitive(clazz)) {
      final Property property = createBaseTypeProperty(clazz);
      property.setName(name);
      return property;
    }
    else if (clazz.isEnum()) {
      final Property property = createEnumProperty(clazz);
      property.setName(name);
      return property;
    }
    else if (clazz.isArray()) {
      final ArrayProperty property = new ArrayProperty();
      property.setName(name);
      return property;
    }
    else if (clazz.getName().equalsIgnoreCase("java.lang.Object")) {
      final ObjectProperty property = new ObjectProperty();
      property.setName(name);
      return property;
    }
    else if (clazz.isAssignableFrom(List.class)) {
      final ArrayProperty property = new ArrayProperty();
      property.setName(name);
      return property;
    }
    else {
      final List<Field> fields = getAnnotatedFields(clazz);
      final ObjectProperty property = new ObjectProperty();
      property.setName(name);
      final Map<String, Property> hash = new HashMap<>();
      for (Field f : fields) {
        hash.put(getFieldName(f), createFieldProperty(level + 1, f));
      }
      property.setProperties(hash);
      return property;
    }
  }

  private Property createEnumProperty(Class clazz) {
    final StringProperty property = new StringProperty();
    final Object[] enumConstants = clazz.getEnumConstants();
    property.setEnum(Stream.of(enumConstants).map(Object::toString).collect(Collectors.toList()));
    return property;
  }

  private Property createFieldProperty(int level, Field field) {
    final String fieldName = getFieldName(field);
    if (level >= MAX_LEVEL) {
      final ObjectProperty property = new ObjectProperty();
      property.setName(fieldName);
      return property;
    }

    final Class<?> clazz = field.getType();
    if (isClassPrimitive(clazz)) {
      final Property property = createBaseTypeProperty(clazz);
      property.setName(fieldName);
      return property;
    }
    else if (clazz.isArray()) {
      final ArrayProperty property = new ArrayProperty();
      property.setName(fieldName);
      return property;
    }
    else if (clazz.isEnum()) {
      final StringProperty property = new StringProperty();
      property.setName(fieldName);
      final Object[] enumConstants = clazz.getEnumConstants();
      property.setEnum(Stream.of(enumConstants).map(Object::toString).collect(Collectors.toList()));
      return property;
    }
    else if (clazz.getName().equalsIgnoreCase("java.lang.Object")) {
      final ObjectProperty property = new ObjectProperty();
      property.setName(fieldName);
      return property;
    }
    else if (clazz.isAssignableFrom(List.class)) {
      if (field.getGenericType() instanceof ParameterizedType) {
        Class<?> listClass = (Class<?>) ((ParameterizedType) field.getGenericType()).getActualTypeArguments()[0];
        final Property property = createArrayProperty(level + 1, listClass);
        property.setName(fieldName);
        return property;
      }
      else {
        final ObjectProperty property = new ObjectProperty();
        property.setName(fieldName);
        return property;
      }
    }
    else {
      final List<Field> fields = getAnnotatedFields(clazz);
      final ObjectProperty property = new ObjectProperty();
      property.setName(fieldName);
      final Map<String, Property> hash = new HashMap<>();
      for (Field f : fields) {
        hash.put(getFieldName(f), createFieldProperty(level + 1, f));
      }
      property.setProperties(hash);
      return property;
    }
  }

  protected Property createBaseTypeProperty(Class<?> clazz) {
    if (byte.class.equals(clazz) || short.class.equals(clazz) || int.class.equals(clazz) || Byte.class.equals(clazz)
      || Short.class.equals(clazz) || Integer.class.equals(clazz)) {
      return new IntegerProperty();
    }
    else if (long.class.equals(clazz) || Long.class.equals(clazz)) {
      return new LongProperty();
    }
    else if (float.class.equals(clazz) || Float.class.equals(clazz)) {
      return new FloatProperty();
    }
    else if (double.class.equals(clazz) || Double.class.equals(clazz)) {
      return new DoubleProperty();
    }
    else if (char.class.equals(clazz) || String.class.equals(clazz)) {
      return new StringProperty();
    }
    else if (boolean.class.equals(clazz) || Boolean.class.equals(clazz)) {
      return new BooleanProperty();
    }
    return null;
  }

  protected <T> ModelImpl createEnumModel(Class clazz) {
    final Object[] enumConstants = clazz.getEnumConstants();
    ModelImpl model = new ModelImpl();
    model.setType("string");
    model.setEnum(Stream.of(enumConstants).map(Object::toString).collect(Collectors.toList()));
    return model;
  }

  protected Property createArrayProperty(int level, Class<?> clazz) {
    final ArrayProperty arrayProperty = new ArrayProperty();
    if (clazz == null) {
      arrayProperty.setItems(new ObjectProperty());
      return arrayProperty;
    }
    final Property baseProperty = createClassProperty(level + 1, "elem", clazz);
    arrayProperty.setItems(baseProperty);
    return arrayProperty;
  }

  private List<Field> getAnnotatedFields(Class<?> clazz) {
    final Field[] fields = clazz.getDeclaredFields();
    return Arrays.stream(fields).filter(f -> f.isAnnotationPresent(JsonProperty.class)).collect(Collectors.toList());
  }

  private String getFieldName(Field field) {
    if (field.isAnnotationPresent(JsonProperty.class)) {
      final JsonProperty annotation = field.getDeclaredAnnotation(JsonProperty.class);
      final String value = annotation.value();
      return value == null ? field.getName() : value;
    }
    return field.getName();
  }

}
