/*
 * $Id$
 */
package reuse.modified.logistic.logic.common;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Classe genrica usada para encapsular cdigos de objetos. O tipo paramtrico
 * T  a classe para a qual o cdigo est sendo criado. Por exemplo Code<User>
 * representa a classe que encapsula um cdigo de usurio. Funciona como um
 * "wrapper" para o cdigo que, na maior parte dos casos,  proveniente do banco
 * de dados.
 * 
 * @param <T> o tipo do objeto que possui esse identificador.
 * 
 * @author mjulia
 */
public class Code<T> implements Serializable, Comparable<Code<T>> {

  /**
   * Identificao da verso da classe, segundo a serializao padro de Java.
   */
  private static final long serialVersionUID = 7617061903509182552L;

  /** O cdigo. */
  private final Object code;

  /** Representao string do cdigo */
  private final String trimmedString;

  /**
   * Constri um cdigo para um objeto da classe parametrizada por T.
   * 
   * @param code o valor do cdigo
   */
  public Code(Object code) {
    this.code = code;
    this.trimmedString = ("" + code).trim();
  }

  /**
   * Obtm o valor do cdigo.
   * 
   * @return o cdigo encapsulado
   */
  public Object getCode() {
    return code;
  }

  /**
   * @return A prpria instncia da classe.
   */
  public Code<T> getObject() {
    return this;
  }

  /**
   * Mtodo utilitrio que gera uma lista com identificadores do tipo T.
   * 
   * @param <T> Tipo do objeto da lista.
   * @param originalList Lista original de objetos.
   * 
   * @see Identifiable
   * @see Code
   * 
   * @return Lista com identificadores de um tipo T.
   */
  public static <T extends Identifiable<T>> List<Code<T>> getCodeList(
    List<T> originalList) {
    List<Code<T>> codeList = new ArrayList<Code<T>>();
    for (T t : originalList) {
      codeList.add(t.getId());
    }
    return codeList;
  }

  /**
   * Compara se esse objeto  igual a outro. Somente so iguais se os cdigos
   * tambm so.
   * 
   * @param other o objeto com o qual esse est sendo comparado.
   * @return verdadeiro, se forem iguais ou falso, caso contrrio.
   */
  @Override
  public boolean equals(Object other) {
    if (other == null) {
      return false;
    }
    if (!(other instanceof Code<?>)) {
      return false;
    }
    Code<?> that = (Code<?>) other;
    if (that.code == null) {
      return this.code == null;
    }
    return trimmedString.equals(that.trimmedString);
  }

  /**
   * Obtm o valor de hashcode desse objeto. Corresponde ao prprio valor hash
   * do cdigo.
   * 
   * @return o hashcode do cdigo
   */
  @Override
  public int hashCode() {
    return trimmedString.hashCode();
  }

  /**
   * Obtm o texto a ser apresentado.
   * 
   * @return texto referente ao cdigo.
   */
  @Override
  public String toString() {
    return trimmedString;
  }

  /**
   * {@inheritDoc}
   * 
   * Os identificadores so transformados em Strings e ordenados da seguinte
   * forma: se so numricos, ordenados em ordem crescente. Caso contrrio,
   * ordenados de forma alfabtica.
   */
  @Override
  public int compareTo(Code<T> other) {
    if (other == null) {
      return 1;
    }
    try {
      int thisInt = Integer.parseInt(trimmedString);
      int otherInt = Integer.parseInt(other.trimmedString);
      return (thisInt < otherInt) ? -1 : (thisInt == otherInt) ? 0 : 1;
    }
    catch (NumberFormatException e) {
      return (trimmedString).compareTo(other.trimmedString);
    }
  }

  /**
   * Converte o tipo do identificador sem warnings. Esse mtodo  necessrio,
   * pois existem diversas converses que so legais, mas no so possveis com
   * um cast normal.<br>
   * <br>
   * 
   * Suponha que voc tem uma varivel que  um Code de SinglePoint<?>. <br>
   * Essa varivel deveria poder ser atribuda a um Code de Point<?> j que
   * SinglePoint<?> herda de Point<?>, mas isso gera um erro. <br>
   * Nem com um cast, isso funciona. <br>
   * Voc pode fazer o cast para Code sem tipo definido, mas a ser necessrio
   * suprimir dois warnings: unchecked e rawtypes. <br>
   * J com o mtodo Code.cast(), voc pode fazer essa converso sem warning nem
   * SuppressWarning.<br>
   * Em alguns casos, quando no for possvel inferir sozinho,  necessrio
   * explicitar o tipo.<br>
   * 
   * <pre>
   * <code>
   * Code<SinglePoint<?>> code1 = getCodeFromSomewhere();
   * //Code<Point<?>> code2 = (Code<Point<?>>) code1; // ERRO!
   * @SuppressWarnings({ "unchecked", "rawtypes" })
   * Code<Point<?>> code2 = (Code) code1; // WARNING!
   * Code<Point<?>> code3 = code1.cast();
   * pointCodeList.add(code1.<Point<?>> cast());
   * </code>
   * </pre>
   * 
   * @param <E> O novo tipo do identificador
   * @return O identificador
   */
  @SuppressWarnings("unchecked")
  public <E> Code<E> cast() {
    return (Code<E>) this;
  }
}
