package busexplorer.panel;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Vector;
import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.SortOrder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.text.Document;

import busexplorer.ApplicationIcons;
import busexplorer.utils.Availability;
import busexplorer.utils.AvailabilityRenderer;
import busexplorer.utils.DateTimeRenderer;
import busexplorer.utils.Language;
import busexplorer.utils.StringVectorRenderer;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;

/**
 * Componente que define um painel com uma {@link SortableTable} e modulariza o
 * uso da mesma (incluso, edio e remoo de elementos da tabela).
 * Opcionalmente  possvel configurar aes relacionadas ao componente.
 * 
 * @author Tecgraf
 * @param <T> Tipo de dado associado a tabela do componente
 */
public class TablePanelComponent<T> extends RefreshablePanel {

  /** Table */
  private SortableTable table;
  /** Painel de scroll da tabela */
  private JScrollPane scrollPane;
  /** Painel de botes */
  private JPanel buttonsPanel;
  /** Boto de adio */
  private JButton addBtn;
  /** Boto de edio */
  private JButton editBtn;
  /** Boto de remoo */
  private JButton removeBtn;
  /** Botes personalizados */
  private List<JButton> othersBtns = new ArrayList<JButton>();
  /** Ao de adio */
  private TablePanelActionInterface<T> addAction;
  /** Ao de edio */
  private TablePanelActionInterface<T> editAction;
  /** Ao de remoo */
  private TablePanelActionInterface<T> removeAction;
  /** Conjunto de aes a serem includas como "outros" */
  private List<TablePanelActionInterface<T>> othersActions =
    new ArrayList<TablePanelActionInterface<T>>();
  /** Indicador se possui algum boto a ser includo na GUI */
  private boolean hasBtns = false;
  /** Indicador se  necessrio painel de filtro sobre a tabela */
  private boolean hasFilter = true;
  /** Indicador se  necessrio painel de botes abaixo da tabela */
  private boolean hasBtnsPanel = true;
  /** Ao de Refresh */
  private TablePanelActionInterface<T> refreshAction;
  /** Campo de filtro */
  private JTextField filterText;
  /** Boto de limpar texto de filtro */
  private JButton clearButton;

  /**
   * Construtor a partir da lista de objetos e um {@link ObjectTableProvider} para controlar a exibio das colunas
   *
   * @param pInfo Lista com os dados da tabela.
   * @param pTableProvider Provedor de dados da tabela.
   * @param actions Conjunto de aes relacionadas ao componente.
   */
  public TablePanelComponent(List<T> pInfo, ObjectTableProvider<T> pTableProvider,
                             List<? extends TablePanelActionInterface<T>> actions) {
    this(new ObjectTableModel<T>(pInfo, pTableProvider), actions, true);
  }

  /**
   * Construtor com {@link ObjectTableModel} personalizado e opo de desativar filtro
   *
   * @param pTableModel Modelo da tabela.
   * @param actions Conjunto de aes relacionadas ao componente.
   * @param hasFilter Deve ser {@code true} caso se queira construir o painel contendo uma
   *                  barra de pesquisa para filtrar os resultados, ou {@code false} caso contrrio.
   *                  Valor padro  {@code true}.
   */
  public TablePanelComponent(ObjectTableModel<T> pTableModel,
                               List<? extends TablePanelActionInterface<T>> actions, boolean hasFilter) {
    this(pTableModel, actions, hasFilter, true);
  }

  /**
   * Construtor com {@link ObjectTableModel} personalizado, opo de desativar filtro e painel de botes
   *
   * @param pTableModel Modelo da tabela.
   * @param actions Conjunto de aes relacionadas ao componente.
   * @param hasFilter Deve ser {@code true} caso se queira construir o painel contendo uma
   *                  barra de pesquisa para filtrar os resultados, ou {@code false} caso contrrio.
   *                  Valor padro  {@code true}.
   * @param hasBtnsPanel Deve ser {@code true} caso se queira construir o painel contendo uma barra
   *                       de botes para as aes, ou {@code false} caso contrrio. Valor padro  {@code true}.
   */
  public TablePanelComponent(ObjectTableModel<T> pTableModel,
                             List<? extends TablePanelActionInterface<T>> actions, boolean hasFilter, boolean hasBtnsPanel) {
    createTable(pTableModel);
    processActions(actions);
    this.hasFilter = hasFilter;
    this.hasBtnsPanel = hasBtnsPanel;
    init();
  }

  /**
   * Cria a tabela do componente.
   * 
   * @param model o modelo da tabela.
   */
  private void createTable(ObjectTableModel<T> model) {
    table = new SortableTable(true);
    table.setModel(model);
    table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    table.sort(0, SortOrder.ASCENDING);
    table.setDefaultRenderer(Vector.class, new StringVectorRenderer());
    table.setDefaultRenderer(Date.class, new DateTimeRenderer());
    table.setDefaultRenderer(Availability.class, new AvailabilityRenderer());
    table.getSelectionModel().addListSelectionListener(
      new TableSelectionListener(table));

    DirectActionsListener l = new DirectActionsListener();
    table.addMouseListener(l);
    table.addKeyListener(l);
    table.adjustSize();
  }

  /**
   * Processa o conjunto de aes associadas ao componente.
   * 
   */
  private void processActions(List<? extends TablePanelActionInterface<T>> actions) {
    for (TablePanelActionInterface<T> action : actions) {
      action.setTablePanelComponent(this);
      switch (action.getActionType()) {
        case ADD:
          addAction = action;
          addAction.setEnabled(addAction.abilityConditions());

          addBtn = new JButton(action);
          addBtn
            .setToolTipText(Language.get(this.getClass(), "add.tooltip"));
          addBtn.setIcon(ApplicationIcons.ICON_ADD_16);
          hasBtns = true;
          break;

        case REMOVE:
          removeAction = action;
          removeAction.setEnabled(false);

          removeBtn = new JButton(action);
          removeBtn.setToolTipText(Language.get(this.getClass(),
            "remove.tooltip"));
          removeBtn.setIcon(ApplicationIcons.ICON_DELETE_16);
          hasBtns = true;
          break;

        case EDIT:
          editAction = action;
          editAction.setEnabled(false);

          editBtn = new JButton(action);
          editBtn.setToolTipText(Language.get(this.getClass(),
            "edit.tooltip"));
          editBtn.setIcon(ApplicationIcons.ICON_EDIT_16);
          hasBtns = true;
          break;

        case REFRESH:
          this.refreshAction = action;
          break;

        case OTHER_ONLY_MULTI_SELECTION:
        case OTHER_MULTI_SELECTION:
        case OTHER_SINGLE_SELECTION:
          // informaes como tooltip e icones devem ser configurados na ao
          othersActions.add(action);
          action.setEnabled(false);
          othersBtns.add(new JButton(action));
          hasBtns = true;
          break;

        default:
          // informaes como tooltip e icones devem ser configurados na ao
          othersActions.add(action);
          othersBtns.add(new JButton(action));
          hasBtns = true;
          break;
      }
    }
  }

  /** Inicializao interna */
  private void init() {
    // Define BorderLayout como o gerenciador padro
    this.setLayout(new BorderLayout());
    if (hasFilter) {
      this.add(getFilterPanel(), BorderLayout.NORTH);
    }
    this.add(getScrollPane(), BorderLayout.CENTER);
    if (hasBtnsPanel && hasBtns) {
      this.add(getButtonsPanel(), BorderLayout.SOUTH);
    }
    this.validate();
    this.setVisible(true);
  }

  /**
   * Constri o painel de filtro da tabela
   * 
   * @return o painel de filtro.
   */
  private JPanel getFilterPanel() {
    final JPanel panel = new JPanel(new GridBagLayout());
    GBC gbc = new GBC(0, 0).west().insets(10, 10, 10, 0);
    final JLabel filterLabel =
      new JLabel(Language.get(this.getClass(), "filter.label"));
    panel.add(filterLabel, gbc);

    filterText = new JTextField();
    gbc = new GBC(1, 0).insets(10).horizontal().filly();
    panel.add(filterText, gbc);
    filterText.setToolTipText(Language
      .get(this.getClass(), "filter.tooltip"));

    clearButton = new JButton(Language.get(this.getClass(), "filter.clear"));
    gbc = new GBC(2, 0).east().insets(10, 0, 10, 10);
    clearButton.setToolTipText(Language.get(this.getClass(),
      "filter.clear.tooltip"));
    clearButton.setIcon(ApplicationIcons.ICON_CLEAR_16);
    panel.add(clearButton, gbc);

    JButton refreshButton =
      new JButton(Language.get(this.getClass(), "refresh"));
    refreshButton.setAction(refreshAction);
    gbc = new GBC(3, 0).east().insets(10, 0, 10, 10);
    refreshButton.setToolTipText(Language.get(this.getClass(),
      "refresh.tooltip"));
    refreshButton.setIcon(ApplicationIcons.ICON_REFRESH_16);
    panel.add(refreshButton, gbc);

    setupFilterControls();
    return panel;
  }

  /**
   * Configura os controles (textfield + boto) para filtragem da tabela com os
   * usurios.
   */
  private void setupFilterControls() {
    // Implementamos um DocumentListener para ativar o filtro quando o contedo
    // do campo for alterado de qualquer forma
    final Document document = filterText.getDocument();
    document.addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        filterTableContent();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        filterTableContent();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        filterTableContent();
      }
    });

    // Queremos que o contedo do filtro seja selecionado inteiro quando o campo
    // ganhar o foco
    filterText.addFocusListener(new FocusListener() {
      @Override
      public void focusGained(FocusEvent e) {
        filterText.selectAll();
      }

      @Override
      public void focusLost(FocusEvent e) {
        filterText.select(0, 0);
      }
    });

    // ao do boto "limpar"
    clearButton.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        filterText.setText("");
        filterTableContent();
      }
    });
  }

  /**
   * Filtra o contedo da tabela a partir do contedo do campo "filtro". O texto
   * do campo  usado como <code>".*texto.*"</code>.
   */
  private void filterTableContent() {
    final String text = filterText.getText();
    if (text.length() > 0) {
      table.setRowFilter(RowFilter.regexFilter(".*" + text + ".*"));
    }
    else {
      table.setRowFilter(null);
    }
  }

  /**
   * Retorna a tabela construdo dentro de um JScrollPane.
   * 
   * @return JScrollPane com todos os componentes do CRUD.
   */
  private JScrollPane getScrollPane() {
    if (scrollPane != null) {
      return scrollPane;
    }
    scrollPane = new JScrollPane(table);
    scrollPane.setViewportView(table);
    scrollPane.setAutoscrolls(true);
    return scrollPane;
  }

  /**
   * Constri o painel de botes.
   * 
   * @return o painel.
   */
  private JPanel getButtonsPanel() {
    List<JButton> toMatch = new ArrayList<>();
    int idx = 0;
    buttonsPanel = new JPanel();
    buttonsPanel.setLayout(new GridBagLayout());
    // separador para alinhar botes a direita
    buttonsPanel.add(new JPanel(), new GBC(idx, 0).horizontal().insets(2));
    idx++;
    if (addBtn != null) {
      buttonsPanel.add(addBtn, new GBC(idx, 0).center().none().insets(2));
      //addBtn.setText(null);
      idx++;
      toMatch.add(addBtn);
    }
    if (editBtn != null) {
      buttonsPanel.add(editBtn, new GBC(idx, 0).center().none().insets(2));
      //editBtn.setText(null);
      idx++;
      toMatch.add(editBtn);
    }
    if (removeBtn != null) {
      buttonsPanel.add(removeBtn, new GBC(idx, 0).center().none().insets(2));
      //removeBtn.setText(null);
      idx++;
      toMatch.add(removeBtn);
    }
    if (!othersBtns.isEmpty()) {
      for (JButton otherBtn : othersBtns) {
        buttonsPanel.add(otherBtn, new GBC(idx, 0).center().none().insets(2));
        idx++;
        toMatch.add(otherBtn);
      }

    }
    GUIUtils.matchPreferredSizes(toMatch.toArray(new JButton[toMatch.size()]));
    return buttonsPanel;
  }

  /**
   * Atualiza as habilitaes das aes cadastradas.
   */
  private void updateActionsAbilities() {
    int[] selectedRows = table.getSelectedRows();
    int length = selectedRows.length;

    if (addAction != null) {
      addAction.setEnabled(addAction.abilityConditions());
    }

    switch (length) {
      case 0:
        if (removeAction != null) {
          removeAction.setEnabled(false);
        }
        if (editAction != null) {
          editAction.setEnabled(false);
        }
        for (TablePanelActionInterface<T> action : othersActions) {
          if (!action.getActionType().equals(ActionType.OTHER)) {
            action.setEnabled(false);
          }
          else {
            action.setEnabled(action.abilityConditions());
          }
        }
        break;

      case 1:
        if (removeAction != null) {
          removeAction.setEnabled(removeAction.abilityConditions());
        }
        if (editBtn != null) {
          editAction.setEnabled(editAction.abilityConditions());
        }
        for (TablePanelActionInterface<T> action : othersActions) {
          switch (action.getActionType()) {
            case OTHER_SINGLE_SELECTION:
            case OTHER_MULTI_SELECTION:
              action.setEnabled(action.abilityConditions());
              break;
            case OTHER_ONLY_MULTI_SELECTION:
              action.setEnabled(false);
              break;
            default:
              action.setEnabled(action.abilityConditions());
              break;
          }
        }
        break;

      default:
        // length > 1
        if (removeAction != null) {
          removeAction.setEnabled(removeAction.abilityConditions());
        }
        if (editAction != null) {
          editAction.setEnabled(false);
        }
        for (TablePanelActionInterface<T> action : othersActions) {
          switch (action.getActionType()) {
            case OTHER_SINGLE_SELECTION:
              action.setEnabled(false);
              break;
            case OTHER_MULTI_SELECTION:
            case OTHER_ONLY_MULTI_SELECTION:
              action.setEnabled(action.abilityConditions());
              break;
            default:
              action.setEnabled(action.abilityConditions());
              break;
          }
        }
        break;
    }
  }

  /**
   * Configura a lista de elementos associados  tabela.
   * 
   * @param objects a lista de elementos.
   */
  public void setElements(List<T> objects) {
    ObjectTableModel<T> model = getTableModel();
    model.setRows(objects);
    table.adjustSize();
  }

  /**
   * Recupera a lista de elementos associados  tabela.
   *
   * @return o conjunto de elementos.
   */
  public List<T> getElements() {
    return getTableModel().getRows();
  }

  /**
   * Recupera o elemento selecionado.
   * 
   * @return o elemento.
   */
  public T getSelectedElement() {
    final int row = table.getSelectedRow();
    if (row < 0) {
      return null;
    }
    int modelRow = table.convertRowIndexToModel(row);
    return getTableModel().getRow(modelRow);
  }

  /**
   * Recupera o conjunto de elementos selecionados.
   * 
   * @return o conjunto de elementos.
   */
  public List<T> getSelectedElements() {
    List<T> selections = new ArrayList<>();
    for (int row : table.getSelectedRows()) {
      int modelRow = table.convertRowIndexToModel(row);
      T object = getTableModel().getRow(modelRow);
      selections.add(object);
    }
    return selections;
  }

  /**
   * Remove o elemento selecionado.
   * 
   * @return o elemento removido.
   */
  public T removeSelectedElement() {
    final int row = table.getSelectedRow();
    if (row < 0) {
      return null;
    }
    int modelRow = table.convertRowIndexToModel(row);
    return getTableModel().remove(modelRow);
  }

  /**
   * Remove o conjunto de elementos selecionados.
   * 
   * @return os elementos removidos.
   */
  public List<T> removeSelectedElements() {
    List<T> removed = new ArrayList<>();
    int[] rows = table.getSelectedRows();
    if (rows.length <= 0) {
      return new ArrayList<>();
    }
    int[] modelRows = new int[rows.length];
    for (int i = 0; i < rows.length; i++) {
      modelRows[i] = table.convertRowIndexToModel(rows[i]);
    }
    removed.addAll(getTableModel().removeAll(modelRows));
    return removed;
  }

  /**
   * Seleciona elemento especificado pelo objeto, e, opcionalmente, o exibe.
   *
   * @param object Objeto a ser selecionado.
   * @param show Indica se o elemento selecionado deve ser exibido.
   */
  public void selectElement(T object, boolean show) {
    int index = this.getTableModel().getRows().indexOf(object);
    if (index != -1) {
      int vIndex = table.convertRowIndexToView(index);
      this.table.getSelectionModel().setSelectionInterval(vIndex, vIndex);
      if (!show) {
        return;
      }
      JViewport viewport = (JViewport)table.getParent();
      Rectangle rectangle = table.getCellRect(vIndex, vIndex, true);
      Point point = viewport.getViewPosition();
      rectangle.setLocation(rectangle.x - point.x, rectangle.y - point.y);
      viewport.scrollRectToVisible(rectangle);
    }
  }

  /**
   * Realiza uma atualizao do contedo apresentado pelo painel, possivelmente
   * realizando uma chamada remota.
   *
   * @param event o evento que dispara a ao.
   */
  @Override
  public void refresh(ActionEvent event) {
    if (this.refreshAction != null) {
      this.refreshAction.actionPerformed(event);
    }
    updateActionsAbilities();
  }

  /**
   * Recupera o modelo da tabela.
   * 
   * @return o modelo da tabela.
   */
  protected ObjectTableModel<T> getTableModel() {
    return (ObjectTableModel<T>) this.table.getModel();
  }

  /**
   * Recupera a tabela deste painel.
   * 
   * @return a tabela do painel.
   */
  protected SortableTable getTable() {
    return this.table;
  }

  /**
   * Listener de seleo da tabela.
   * 
   * @author Tecgraf
   */
  class TableSelectionListener implements ListSelectionListener {

    /**
     * Referncia para a tabela do componente.
     */
    private JTable table;

    /**
     * Construtor.
     * 
     * @param table a tabela sendo observada.
     */
    public TableSelectionListener(JTable table) {
      this.table = table;
    }

    /** {@inheritDoc} */
    @Override
    public void valueChanged(ListSelectionEvent e) {
      updateActionsAbilities();
    }
  }

  /**
   * Inclui listener de mouse na tabela do painel.
   * 
   * @param listener o listener a ser includo.
   */
  public void addTableMouseListener(MouseListener listener) {
    table.addMouseListener(listener);
  }

  /**
   * Inclui listener de teclado na tabela do painel.
   * 
   * @param listener o listener a ser includo.
   */
  public void addTableKeyListener(KeyListener listener) {
    table.addKeyListener(listener);
  }

  /**
   * Listener para eventos de mouse e teclado que acionam as aes fornecidas
   * pelo CRUD, sem que o usurio tenha que utilizar seus botes
   * correspondentes.
   */
  class DirectActionsListener extends MouseAdapter implements KeyListener {

    /** {@inheritDoc} */
    @Override
    public void mouseClicked(MouseEvent e) {
      if (e.getClickCount() == 2) {
        int selectedRow = table.getSelectedRow();
        int selectedColumn = table.getSelectedColumn();
        if (table.isCellEditable(selectedRow, selectedColumn)) {
          /* Caso a clula seja editvel, inibe a ao de edio via boto */
        }
        else if (editBtn != null && editBtn.isEnabled()) {
          editBtn.doClick();
        }
      }
    }

    /** {@inheritDoc} */
    @Override
    public void keyPressed(KeyEvent e) {
    }

    /** {@inheritDoc} */
    @Override
    public void keyReleased(KeyEvent e) {
      /** Tecla Delete */
      if (e.getKeyCode() == KeyEvent.VK_DELETE) {
        int selectedRow = table.getSelectedRow();
        int selectedColumn = table.getSelectedColumn();
        if (table.isCellEditable(selectedRow, selectedColumn)) {
          /* Caso a clula seja editvel, inibe a ao de edio via teclado */
        }
        else if (removeBtn != null && removeBtn.isEnabled()) {
          removeBtn.doClick();
        }
      }
    }

    /** {@inheritDoc} */
    @Override
    public void keyTyped(KeyEvent e) {
    }

  }

}
