package migration.rest;

import migration.dao.JobDAO;
import migration.utils.Log;
import migration.utils.PrettyStatistics;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class CSBASERestMigration {
  Properties config;
  private APICommunication api;
  private PrettyStatistics counting;
  private PrettyStatistics projects;

  public CSBASERestMigration(Properties config) {
    this.config = config;
    counting = new PrettyStatistics(new HashMap<>(), config);
    api = new APICommunication(counting, config);
    projects = new PrettyStatistics(new HashMap<>(), config);
  }

  public void execute() {

    JobDAO jobDao = new JobDAO(this.config);
    // PROJECTS
    List<String> projectList = getProjectList();

    if (!projectList.isEmpty()) {
      counting.set("MIGRATED", 0);
      for (int i = 0; i < projectList.size(); i++) {

        counting.plusOne("TOTAL_PROJECTS");
        JSONObject fullJobJSONObj = null;
        try {
          String projectId = projectList.get(i);
          Log.print(
              String.format("Project ID: %s (%d of %d)", projectId, i + 1, projectList.size()));

          projects.set(projectId + "_TOTAL_JOBS", 0);

          // JOBS
          String encodedProjectId =
              Base64.getEncoder().encodeToString(projectId.getBytes(StandardCharsets.UTF_8));

          JSONArray jobsIdsJSONArray = api.getJobsIdsFromProject(encodedProjectId);
          for (int j = 0; j < jobsIdsJSONArray.length(); j++) {
            String jobId = jobsIdsJSONArray.getString(j);

            counting.plusOne("TOTAL_JOBS");
            projects.plusOne(projectId + "_TOTAL_JOBS");
            fullJobJSONObj = api.getFullJob(jobId);

            if (fullJobJSONObj == null) {

              Log.print(String.format("Erro migrando job %s", jobId));
              continue;
            }

            JSONObject jobJSONObj = fullJobJSONObj.getJSONObject("job");

            Log.print(
                String.format(
                    "--- JOB ID: %s (%d of %d for project %s)",
                    jobId, j + 1, jobsIdsJSONArray.length(), projectId));

            if (jobDao.jobExists(jobId)) {
              Log.print(String.format("--- JOB ID: %s already migrated. Skipping ", jobId));
              counting.plusOneError("JOB_ALREADY_MIGRATED", jobJSONObj);
              continue;
            }

            List<JSONObject> algorithms = api.getAlgorithmsFromFullJob(fullJobJSONObj);
            JSONArray nodesJSONArray = fullJobJSONObj.getJSONArray("nodes");

            if (algorithms != null) {
              if (jobDao.insertJob(jobJSONObj, algorithms, nodesJSONArray)) {
                counting.plusOne("MIGRATED");
                projects.plusOne(projectId + "_TOTAL_MIGRATED_JOBS");
              } else {
                counting.plusOneError("ERROR_DB_RESTRICTION", jobJSONObj);
                projects.plusOne(projectId + "_TOTAL_ERROR_DB_RESTRICTION_JOBS");
              }
            }
          }
        } catch (JSONException e) {
          counting.plusOneError("error_JSONException_function_execute_project", fullJobJSONObj);
          Log.print(e);
        }
      }
    }
    Log.print(counting);
    Log.print(projects);
  }

  // Projects
  private List<String> getProjectList() {
    List<String> projectList = getProjectListFromProperties();
    if (projectList.isEmpty()) {
      projectList = getProjectListFromServer();
    }

    return projectList;
  }

  private List<String> getProjectListFromProperties() {

    if (config.containsKey("projectList")) {
      return Arrays.stream(config.getProperty("projectList").split(","))
          .filter(Predicate.not(String::isBlank))
          .collect(Collectors.toList());
    } else {
      return Collections.EMPTY_LIST;
    }
  }

  private List<String> getProjectListFromServer() {
    JSONArray projectJSONArray = api.getProjects();
    if (projectJSONArray.length() == 0) {
      return Collections.EMPTY_LIST;
    } else {
      List<String> projectList = new ArrayList<>();
      for (int i = 0; i < projectJSONArray.length(); i++) {
        JSONObject projectJSONObj;
        try {
          projectJSONObj = projectJSONArray.getJSONObject(i);
          String projectId = new String(Base64.getDecoder().decode(projectJSONObj.getString("id")));
          Log.print("Project ID: " + projectId);
          projectList.add(projectId);
        } catch (JSONException e) {
          e.printStackTrace();
        }
      }
      return projectList;
    }
  }

  // Jobs
  private List<String> getJobsFromProject(String projectId) {

    return Collections.EMPTY_LIST;
  }
}
