package migration.rest;

import migration.utils.Log;
import migration.utils.PrettyStatistics;
import migration.utils.Utils;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

public class APICommunication {
  private Properties config;
  private PrettyStatistics counting;
  private JSONObject token;

  public APICommunication(PrettyStatistics counting, Properties config) {
    this.config = config;
    this.counting = counting;
    this.token = this.getToken();
  }

  private String getEndPoint(String endpoint_key) {
    return config.getProperty("api_base_url") + config.getProperty(endpoint_key);
  }

  private JSONObject getToken() {
    JSONObject token = null;
    try {
      URL url = new URL(getEndPoint("authentication_endpoint"));
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setDoOutput(true);
      conn.setRequestMethod("POST");
      conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
      String input =
          "login="
              + config.getProperty("authentication_user")
              + "&password="
              + config.getProperty("authentication_password");
      OutputStream os = conn.getOutputStream();
      os.write(input.getBytes());
      os.flush();
      if (conn.getResponseCode() != HttpURLConnection.HTTP_OK) {
        System.out.println("Failed : HTTP error code : " + conn.getResponseCode());
        System.out.println("Failed : ERROR_getToken_HTTP_error_code : " + conn.getResponseCode());
        counting.plusOneError("ERROR_getToken_HTTP_error_code_" + conn.getResponseCode(), null);
        conn.disconnect();
        System.exit(1);
      } else {
        BufferedReader br = new BufferedReader(new InputStreamReader((conn.getInputStream())));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = br.readLine()) != null) {
          response.append(inputLine);
        }
        token = new JSONObject(response.toString());
        conn.disconnect();
      }
    } catch (IOException | JSONException e) {
      System.out.println("Failed : ERROR_getToken_HTTP_error");
      e.printStackTrace();
      System.exit(1);
    }
    return token;
  }

  public JSONArray getProjects() {
    JSONArray projects = null;
    try {
      URL url = new URL(getEndPoint("projects_endpoint"));
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setRequestMethod("GET");
      conn.setRequestProperty("Accept", "application/json");
      conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
      conn.setRequestProperty(
          "Authorization", token.getString("tokenType") + token.getString("accessToken"));
      if (conn.getResponseCode() != 200) {
        Log.print("URL: " + url);
        Log.print("Failed : HTTP error code : " + conn.getResponseCode());
        counting.plusOneError("ERROR_getProjects_HTTP_error_code_" + conn.getResponseCode(), null);
        conn.disconnect();
        return null;
      } else {
        BufferedReader br = new BufferedReader(new InputStreamReader((conn.getInputStream())));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = br.readLine()) != null) {
          response.append(inputLine);
        }
        projects = new JSONArray(response.toString());
        conn.disconnect();
      }
    } catch (JSONException | IOException e) {
      e.printStackTrace();
    }
    return projects;
  }

  public JSONArray getJobsIdsFromProject(String projectId) {
    JSONArray jobsIdsJSONArray = null;
    try {
      URL url = new URL(getEndPoint("jobs_ids_endpoint").replace("{{projectId}}", projectId));
      System.out.println(url);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setRequestMethod("GET");
      conn.setRequestProperty("Accept", "application/json");
      conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
      conn.setRequestProperty(
          "Authorization", token.getString("tokenType") + token.getString("accessToken"));
      if (conn.getResponseCode() != 200) {
        Log.print("Failed : HTTP error code : " + conn.getResponseCode());
        counting.plusOneError(
            "ERROR_getJobsFromProject_HTTP_error_code_" + conn.getResponseCode(), null);
        conn.disconnect();
        return null;
      } else {
        BufferedReader br = new BufferedReader(new InputStreamReader((conn.getInputStream())));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = br.readLine()) != null) {
          response.append(inputLine);
        }
        jobsIdsJSONArray = new JSONArray(response.toString());
        conn.disconnect();
      }
    } catch (IOException | DateTimeParseException | JSONException e) {
      e.printStackTrace();
    }
    return jobsIdsJSONArray;
  }

  public JSONObject getFullJob(String jobId) {

    JSONObject jobJSONObject = null;
    try {
      URL url = new URL(getEndPoint("job_full_endpoint").replace("{{jobId}}", jobId));
      System.out.println(url);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setRequestMethod("GET");
      conn.setRequestProperty("Accept", "application/json");
      conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
      conn.setRequestProperty(
          "Authorization", token.getString("tokenType") + token.getString("accessToken"));
      if (conn.getResponseCode() != 200) {
        Log.print("Failed : HTTP error code : " + conn.getResponseCode());
        counting.plusOneErrorWithJobIdOnly(
            "ERROR_getFullJob_HTTP_error_code_" + conn.getResponseCode(), jobId);
        conn.disconnect();
        return null;
      } else {
        BufferedReader br = new BufferedReader(new InputStreamReader((conn.getInputStream())));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = br.readLine()) != null) {
          response.append(inputLine);
        }
        jobJSONObject = new JSONObject(response.toString());
        conn.disconnect();
      }
    } catch (IOException | DateTimeParseException | JSONException e) {
      e.printStackTrace();
    }

    testEndTime(jobJSONObject);
    return jobJSONObject;
  }

  /**
   * Método criado como "workaround" ao bug da do atributo de endTime do CSBASE. Para não inserir
   * uma informação inconsistente no banco, e na impossibilidade de correção imediata do bug, o time
   * orientou o cálculo do endTime do job a partir da data de submissão (submissionTime) + tempo de
   * processamento (wallclocktime)
   *
   * @return lista de jobs no formato JSON com endTimes calculados a partir do (submissionTime) +
   *     (wallclocktime)
   */
  private void testEndTime(JSONObject fullJobJSONObj) {
    try {
      if (config.containsKey("use_job_walltime_to_calc_endtime")
          && Boolean.valueOf(config.getProperty("use_job_walltime_to_calc_endtime")).equals(true)) {
        if (fullJobJSONObj != null) {
          JSONObject jobJSONObj = fullJobJSONObj.getJSONObject("job");
          if (jobJSONObj.has("submissionTime") && jobJSONObj.has("wallclockTime")) {
            String submissionTime = jobJSONObj.getString("submissionTime");
            int wallclockTime = Integer.valueOf(jobJSONObj.getString("wallclockTime"));
            jobJSONObj.put("endTime", Utils.addSecondToTimestamp(submissionTime, wallclockTime));
          } else {
            jobJSONObj.put("endTime", jobJSONObj.getString("submissionTime"));
          }
        }
      }
    } catch (JSONException e) {
      e.printStackTrace();
    }
  }

  public List<JSONObject> getAlgorithmsFromFullJob(JSONObject fullJobJSONObj) throws JSONException {

    JSONObject jobJSONObj = fullJobJSONObj.getJSONObject("job");
    if (jobJSONObj.getString("jobType").equalsIgnoreCase("flow")) {
      return buildAlgorithmsFromFlow(jobJSONObj, fullJobJSONObj.getJSONObject("parameters"));
    } else {
      String algorithmId = jobJSONObj.getString("algorithmId");
      String algorithmVersion = jobJSONObj.getString("algorithmVersion");
      JSONObject paramsFromAlgorithm =
          getConfiguratorFromAlgorithm(algorithmId, algorithmVersion, jobJSONObj);
      if (paramsFromAlgorithm == null) {
        return null;
      } else {
        return buildAlgorithmForJob(
            jobJSONObj, paramsFromAlgorithm, fullJobJSONObj.getJSONObject("parameters"));
      }
    }
  }

  private List<JSONObject> buildAlgorithmsFromFlow(
      JSONObject jobJSONObj, JSONObject algorithmsFromJob) throws JSONException {
    List<JSONObject> output = new ArrayList<>();
    Iterator<String> keys = algorithmsFromJob.keys();
    while (keys.hasNext()) {
      JSONArray paramsFromJob = new JSONArray();
      String key = keys.next();
      JSONObject algorithmData = algorithmsFromJob.getJSONObject(key);

      JSONObject paramsFromAlgorithm =
          getConfiguratorFromAlgorithm(
              algorithmData.getString("id"), algorithmData.getString("version"), jobJSONObj);
      if (paramsFromAlgorithm != null) {
        JSONObject params = algorithmData.getJSONObject("parameters");
        Iterator<String> paramsKeys = params.keys();
        while (paramsKeys.hasNext()) {
          String paramKey = paramsKeys.next();
          JSONObject configParam = getParamConfig(paramsFromAlgorithm, paramKey, jobJSONObj);

          JSONObject param = new JSONObject();
          param.put("param_id", paramKey);

          param.put("param_type", params.getJSONObject(paramKey).getString("type"));
          param.put("param_label", configParam.getString("label"));
          param.put("param_value", params.getJSONObject(paramKey).getString("values"));
          paramsFromJob.put(param);
        }
        algorithmData.put("params", paramsFromJob);
        output.add(algorithmData);
      }
    }

    return output;
  }

  private List<JSONObject> buildAlgorithmForJob(
      JSONObject jobJSONObj, JSONObject paramsFromAlgorithm, JSONObject paramsFromJobDict)
      throws JSONException {
    String algorithmId = jobJSONObj.getString("algorithmId");
    String algorithmName = jobJSONObj.getString("algorithmName");
    String algorithmVersion = jobJSONObj.getString("algorithmVersion");

    List<JSONObject> output = new ArrayList<>();
    JSONObject algorithmData = new JSONObject();
    algorithmData.put("id", algorithmId);
    algorithmData.put("name", algorithmName);
    algorithmData.put("version", algorithmVersion);

    JSONArray paramsFromJob = new JSONArray();

    Iterator<String> keys = paramsFromJobDict.keys();
    while (keys.hasNext()) {
      String key = keys.next();
      JSONObject configParam = getParamConfig(paramsFromAlgorithm, key, jobJSONObj);
      if (configParam != null) {
        JSONObject param = new JSONObject();
        param.put("param_id", key);
        param.put("param_type", paramsFromJobDict.getJSONObject(key).getString("type"));
        param.put("param_label", configParam.getString("label"));
        param.put("param_value", paramsFromJobDict.getJSONObject(key).getString("values"));
        paramsFromJob.put(param);
      }
    }
    algorithmData.put("params", paramsFromJob);

    output.add(algorithmData);

    return output;
  }

  private JSONObject getConfiguratorFromAlgorithm(
      String algorithmId, String algorithmVersion, JSONObject jobJSONObject) throws JSONException {
    JSONObject configFromAlgorithm = null;
    try {
      String urlString = getEndPoint("configuration_endpoint");
      urlString = urlString.replace("{{algorithmId}}", algorithmId);
      urlString = urlString.replace("{{algorithmVersion}}", algorithmVersion);
      URL url = new URL(urlString);
      HttpURLConnection conn = (HttpURLConnection) url.openConnection();
      conn.setRequestMethod("GET");
      conn.setRequestProperty("Accept", "application/json");
      conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
      conn.setRequestProperty(
          "Authorization", token.getString("tokenType") + token.getString("accessToken"));
      if (conn.getResponseCode() == 200) {
        BufferedReader br = new BufferedReader(new InputStreamReader((conn.getInputStream())));
        String inputLine;
        StringBuilder response = new StringBuilder();
        while ((inputLine = br.readLine()) != null) {
          response.append(inputLine);
        }
        if (response.length() > 0) {
          configFromAlgorithm = new JSONObject(response.toString());
        }
      } else {
        Log.print(
            algorithmId
                + " GET getConfiguratorFromAlgorithm FAILED : HTTP error code : "
                + conn.getResponseCode());
        counting.plusOneError(
            "ERROR_getConfiguratorFromAlgorithm_HTTP_error_code_" + conn.getResponseCode(),
            jobJSONObject);
      }
      conn.disconnect();
    } catch (IOException e) {
      Log.print(e.toString());
    }
    return configFromAlgorithm;
  }

  private JSONObject getParamConfig(
      JSONObject configuration, String paramKey, JSONObject jobJSONObject) throws JSONException {
    JSONArray configArray = configuration.getJSONArray("groups");
    for (int i = 0; i < configArray.length(); i++) {
      JSONArray configGroup = configArray.getJSONObject(i).getJSONArray("parameters");
      for (int j = 0; j < configGroup.length(); j++) {
        if (configGroup.getJSONObject(j).getString("id").equals(paramKey)) {
          return configGroup.getJSONObject(j);
        }
      }
    }
    counting.plusOneError("ERROR_configuration_not_found", jobJSONObject);
    return null;
  }
}
