package br.pucrio.tecgraf.soma.job.application.service.job;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import br.pucrio.tecgraf.soma.job.Constants;
import br.pucrio.tecgraf.soma.job.JobExecutingEvent;
import br.pucrio.tecgraf.soma.job.JobFinishedEvent;
import br.pucrio.tecgraf.soma.job.JobInitEvent;
import br.pucrio.tecgraf.soma.job.JobProgressEvent;
import br.pucrio.tecgraf.soma.job.JobRescheduledEvent;
import br.pucrio.tecgraf.soma.job.JobScheduledEvent;
import br.pucrio.tecgraf.soma.job.JobStageInEvent;
import br.pucrio.tecgraf.soma.job.JobStageOutEvent;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.ExecutingJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.FinishedJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.InitJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.JobProgressBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.RescheduledJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.ScheduledJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.StageInJobBuilder;
import br.pucrio.tecgraf.soma.job.application.service.job.builder.StageOutJobBuilder;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Repository;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;

@Service
public class JobService {

	private final Logger logger = LoggerFactory.getLogger(JobService.class);

	private Repository<Job, JPASpecification<Job>> jobRepository;

	@Autowired
	public JobService(Repository<Job, JPASpecification<Job>> jobRepository) {
		this.jobRepository = jobRepository;
	}

	@Transactional
	public void scheduleJob(JobScheduledEvent scheduledEvent) throws IOException {
    try {
      logEvent(scheduledEvent.getJobId(), Constants.JOB_SCHEDULED_EVENT, scheduledEvent.getEventId());
      Job job = new Job();

      if(scheduledEvent.getDependencyIds() == null) {
        new ScheduledJobBuilder().build(job, scheduledEvent);
      } else {
        Map<String, Job> dependencies = new HashMap<>();
        for(String dependencyId : scheduledEvent.getDependencyIds()) {
          dependencies.put(dependencyId, jobRepository.first(new JobByIdSpecification(dependencyId)));
        }
        new ScheduledJobBuilder().build(job, scheduledEvent, dependencies);
      }

      jobRepository.add(job);
    } catch (Exception e) {
      throw createIOException(
          scheduledEvent.getJobId(),
          Constants.JOB_SCHEDULED_EVENT,
          scheduledEvent.getEventId(),
          e);
    }
  }

  @Transactional
	public void rescheduleJob(JobRescheduledEvent rescheduledEvent) throws IOException {
    try {
      logEvent(rescheduledEvent.getJobId(), Constants.JOB_RESCHEDULED_EVENT, rescheduledEvent.getEventId());
      Job job = findJob(rescheduledEvent.getJobId());
      new RescheduledJobBuilder().build(job, rescheduledEvent);
    } catch (Exception e) {
      throw createIOException(
          rescheduledEvent.getJobId(),
          Constants.JOB_RESCHEDULED_EVENT,
          rescheduledEvent.getEventId(),
          e);
    }
  }

  @Transactional
	public void initJob(JobInitEvent jobInitEvent) throws IOException {
		logEvent(jobInitEvent.getJobId(), Constants.JOB_INIT_EVENT, jobInitEvent.getEventId());
		try {
			Job job = findJob(jobInitEvent.getJobId());
			new InitJobBuilder().build(job, jobInitEvent);
    } catch (Exception e) {
      throw createIOException(
        jobInitEvent.getJobId(),
          Constants.JOB_INIT_EVENT,
          jobInitEvent.getEventId(),
          e);
    }
	}

  @Transactional
	public void stageJobIn(JobStageInEvent stageInEvent) throws IOException {
		logEvent(stageInEvent.getJobId(), Constants.JOB_STAGEIN_EVENT, stageInEvent.getEventId());
		try {
			Job job = findJob(stageInEvent.getJobId());
			new StageInJobBuilder().build(job, stageInEvent);
    } catch (Exception e) {
      throw createIOException(
          stageInEvent.getJobId(),
          Constants.JOB_STAGEIN_EVENT,
          stageInEvent.getEventId(),
          e);
    }
	}

  @Transactional
  public void stageJobOut(JobStageOutEvent stageOutEvent) throws IOException {
		logEvent(stageOutEvent.getJobId(), Constants.JOB_STAGEOUT_EVENT, stageOutEvent.getEventId());
		try {
			Job job = findJob(stageOutEvent.getJobId());
			new StageOutJobBuilder().build(job, stageOutEvent);
    } catch (Exception e) {
      throw createIOException(
          stageOutEvent.getJobId(),
          Constants.JOB_STAGEOUT_EVENT,
          stageOutEvent.getEventId(),
          e);
    }
	}

  @Transactional
  public void executeJob(JobExecutingEvent executingEvent) throws IOException {
		logEvent(executingEvent.getJobId(), Constants.JOB_EXECUTING_EVENT, executingEvent.getEventId());
		try {
			Job job = findJob(executingEvent.getJobId());
			new ExecutingJobBuilder().build(job, executingEvent);
    } catch (Exception e) {
      throw createIOException(
          executingEvent.getJobId(),
          Constants.JOB_EXECUTING_EVENT,
          executingEvent.getEventId(),
          e);
    }
	}

  @Transactional
  public void finishJob(JobFinishedEvent finishedEvent) throws IOException {
		logEvent(finishedEvent.getJobId(), Constants.JOB_FINISHED_EVENT, finishedEvent.getEventId());
		try {
			Job job = findJob(finishedEvent.getJobId());
			new FinishedJobBuilder().build(job, finishedEvent);
    } catch (Exception e) {
      throw createIOException(
          finishedEvent.getJobId(),
          Constants.JOB_FINISHED_EVENT,
          finishedEvent.getEventId(),
          e);
    }
	}

  @Transactional
  public void progressUpdate(JobProgressEvent progressEvent) throws IOException {
		logEvent(progressEvent.getJobId(), Constants.JOB_PROGRESS_EVENT, progressEvent.getEventId());
		try {
			Job job = findJob(progressEvent.getJobId());
			new JobProgressBuilder().build(job, progressEvent);
    } catch (Exception e) {
      throw createIOException(
        progressEvent.getJobId(),
          Constants.JOB_PROGRESS_EVENT,
          progressEvent.getEventId(),
          e);
    }
	}

  private Job findJob(String jobId) {
    Job job = jobRepository.first(new JobByIdSpecification(jobId));
    if (job == null) {
      throw new RuntimeException("Cannot find a job with JobId= " + jobId + ".");
    }
    return job;
  }

	private void logEvent(String jobId, String eventType, String eventId){
		logger.info("[Job={}, event type={}, event id={}]", jobId, eventType, eventId);
	}

  private IOException createIOException(
      String jobId, String eventType, String eventId, Exception e) {
    return new IOException(
        MessageFormat.format(
            "Error processing event type {0} for job {1}. [event id={2}].",
            eventType,
            jobId,
            eventId),
        e);
  }
}
