package br.pucrio.tecgraf.soma.job.infrastructure.persistence.message;

import static org.apache.kafka.clients.consumer.ConsumerConfig.AUTO_OFFSET_RESET_CONFIG;
import static org.apache.kafka.clients.consumer.ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG;
import static org.apache.kafka.clients.consumer.ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG;
import static org.apache.kafka.clients.consumer.ConsumerConfig.GROUP_ID_CONFIG;
import static org.apache.kafka.clients.consumer.ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG;
import static org.apache.kafka.clients.consumer.ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG;
import java.io.IOException;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Observable;
import java.util.Properties;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;
import org.springframework.messaging.MessagingException;
import org.springframework.stereotype.Service;
import br.pucrio.tecgraf.soma.job.JobHistoryEvent;
import br.pucrio.tecgraf.soma.job.SomaJobHistoryInfo;
import br.pucrio.tecgraf.soma.job.application.configuration.Constants.Config;
import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import io.confluent.kafka.serializers.KafkaAvroDeserializer;
import io.confluent.kafka.serializers.KafkaAvroDeserializerConfig;
import io.confluent.kafka.serializers.KafkaAvroSerializerConfig;

@Service
public class JobHistoryEventReader extends Observable {

  private final Logger logger = LoggerFactory.getLogger(SomaJobHistoryInfo.class);

  private final String KAFKA_SERVER;
  private final String KAFKA_SCHEMA_REGISTRY_URL;
  private final String KAFKA_TOPIC;
  private final String KAFKA_GROUP;

  @Autowired
  public JobHistoryEventReader(ServiceConfiguration serviceConfiguration) {
    this.KAFKA_SERVER =
        serviceConfiguration.getValue(Config.KAFKA_SERVER_ADDRESS.option.getLongName());
    this.KAFKA_SCHEMA_REGISTRY_URL =
        serviceConfiguration.getValue(Config.KAFKA_SCHEMA_REGISTRY_URL.option.getLongName());
    this.KAFKA_TOPIC = serviceConfiguration.getValue(Config.KAFKA_TOPIC.option.getLongName());
    this.KAFKA_GROUP =
        serviceConfiguration.getValue(Config.KAFKA_CONSUMER_GROUP.option.getLongName());
  }

  @EventListener(ApplicationReadyEvent.class)
  public void run() throws IOException, InterruptedException {
    Consumer<String, JobHistoryEvent> consumer =
        buildKafkaConsumer(KAFKA_SERVER, KAFKA_SCHEMA_REGISTRY_URL, KAFKA_TOPIC, KAFKA_GROUP);
    while (!Thread.currentThread().isInterrupted()) {
      readRecords(consumer);
    }
  }

  protected void readRecords(Consumer<String, JobHistoryEvent> consumer)
      throws InterruptedException, IOException {
    ConsumerRecords<String, JobHistoryEvent> records = consumer.poll(Duration.ofMillis(100));
    logger.debug("Got {} records from Kafka", records.count());
    for (TopicPartition partition : records.partitions()) {
      List<ConsumerRecord<String, JobHistoryEvent>> partitionRecords = records.records(partition);
      for (ConsumerRecord<String, JobHistoryEvent> record : partitionRecords) {
        readRecord(consumer, partition, partitionRecords, record);
      }
    }
  }

  private void readRecord(Consumer<String, JobHistoryEvent> consumer, TopicPartition partition,
      List<ConsumerRecord<String, JobHistoryEvent>> partitionRecords,
      ConsumerRecord<String, JobHistoryEvent> record) throws InterruptedException, IOException {
    try {
      logger.info("Job {} at Kafka topic {}:{}:{}:{}", record.key(), record.topic(),
          record.value(), partition.partition(), record.offset());
      setChanged();
      notifyObservers(record.value());
    } catch (MessagingException me) {
      logger.error("Error trying send message to job-info", me);
    } catch (Exception e) {
      logger.error("Unrecoverable error.", e);
    } finally {
      consumer.commitSync();
    }
  }

  protected Consumer<String, JobHistoryEvent> buildKafkaConsumer(String kafkaServer,
      String schemaRegistryUrl, String topic, String group) {
    Consumer<String, JobHistoryEvent> consumer =
        new KafkaConsumer<>(buildProperties(kafkaServer, schemaRegistryUrl, topic, group));
    consumer.subscribe(Collections.singletonList(topic));
    return consumer;
  }

  protected Properties buildProperties(String kafkaServer, String schemaRegistryUrl, String topic,
      String group) {
    Properties properties = new Properties();
    // kafka bootstrap server
    properties.setProperty(BOOTSTRAP_SERVERS_CONFIG, kafkaServer);
    properties.setProperty(KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class.getName());
    properties.setProperty(VALUE_DESERIALIZER_CLASS_CONFIG, KafkaAvroDeserializer.class.getName());
    properties.setProperty(KafkaAvroSerializerConfig.SCHEMA_REGISTRY_URL_CONFIG, schemaRegistryUrl);
    properties.setProperty(KafkaAvroDeserializerConfig.SPECIFIC_AVRO_READER_CONFIG, "true");

    properties.setProperty(GROUP_ID_CONFIG, group);
    properties.setProperty(ENABLE_AUTO_COMMIT_CONFIG, "false");
    // properties.setProperty(AUTO_COMMIT_INTERVAL_MS_CONFIG, "1000");
    properties.setProperty(AUTO_OFFSET_RESET_CONFIG, "latest");
    return properties;
  }


}
