package br.pucrio.tecgraf.soma.job.application.endpoint;

import java.io.IOException;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import br.pucrio.tecgraf.soma.job.application.configuration.SocketHandler;
import br.pucrio.tecgraf.soma.job.application.configuration.WebSocketConfig;
import org.apache.avro.specific.SpecificRecordBase;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Component;
import br.pucrio.tecgraf.soma.job.Algorithm;
import br.pucrio.tecgraf.soma.job.Flow;
import br.pucrio.tecgraf.soma.job.JobExecutingEvent;
import br.pucrio.tecgraf.soma.job.JobFinishedEvent;
import br.pucrio.tecgraf.soma.job.JobHistoryEvent;
import br.pucrio.tecgraf.soma.job.JobRescheduledEvent;
import br.pucrio.tecgraf.soma.job.JobScheduledEvent;
import br.pucrio.tecgraf.soma.job.JobStageInEvent;
import br.pucrio.tecgraf.soma.job.JobStageOutEvent;
import br.pucrio.tecgraf.soma.job.application.configuration.Constants.Config;
import br.pucrio.tecgraf.soma.job.domain.AlgorithmSpecificationVO;
import br.pucrio.tecgraf.soma.job.domain.JobEventVO;
import br.pucrio.tecgraf.soma.job.domain.StatusTypeVO;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.message.JobHistoryEventReader;
import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import org.springframework.web.socket.TextMessage;

@Component
public class JobEventWebSocket implements Observer {

  private final String BROKER_PREFIX_TYPE;
  private final String TOPIC_NAME;
  private JobHistoryEventReader jobHistoryEventReader;
  private WebSocketConfig webSocketConfig;

  @Autowired
  public JobEventWebSocket(JobHistoryEventReader jobHistoryEventReader, WebSocketConfig webSocketConfig, ServiceConfiguration serviceConfiguration) {
    this.BROKER_PREFIX_TYPE =
        serviceConfiguration.getValue(Config.WEBSOCKET_PREFIX_TYPE.option.getLongName());
    this.TOPIC_NAME =
        serviceConfiguration.getValue(Config.WEBSOCKET_TOPIC_NAME.option.getLongName());
    this.jobHistoryEventReader = jobHistoryEventReader;
    this.jobHistoryEventReader.addObserver(this);
    this.webSocketConfig = webSocketConfig;
  }

  public WebSocketConfig getWebSocketConfig() {
    return webSocketConfig;
  }

  @Override
  public void update(Observable observable, Object arg) {
    SpecificRecordBase record = (SpecificRecordBase) ((JobHistoryEvent) arg).getEvent();
    JobEventVO jobEvent = null;
    if (record instanceof JobStageInEvent) {
      jobEvent = createJobEvent((JobStageInEvent) record);
    } else if (record instanceof JobStageOutEvent) {
      jobEvent = createJobEvent((JobStageOutEvent) record);
    } else if (record instanceof JobExecutingEvent) {
      jobEvent = createJobEvent((JobExecutingEvent) record);
    } else if (record instanceof JobRescheduledEvent) {
      jobEvent = createJobEvent((JobRescheduledEvent) record);
    } else if (record instanceof JobScheduledEvent) {
      jobEvent = createJobEvent((JobScheduledEvent) record);
    } else if (record instanceof JobFinishedEvent) {
      jobEvent = createJobEvent((JobFinishedEvent) record);
    }
    if (jobEvent != null) {
      try {
    	webSocketConfig.getSocketHandler().sendMessage(jobEvent.toJson());
      } catch (InterruptedException e) {
        e.printStackTrace();
      } catch (IOException e) {
          e.printStackTrace();
      }
    }
  }

  private JobEventVO createJobEvent(JobStageInEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.UPDATE);
    jobEvent.setState(StatusTypeVO.UPLOADING.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }

  private JobEventVO createJobEvent(JobStageOutEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.UPDATE);
    jobEvent.setState(StatusTypeVO.DOWNLOADING.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }

  private JobEventVO createJobEvent(JobExecutingEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.UPDATE);
    jobEvent.setState(StatusTypeVO.EXECUTING.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setExecutionMachine(e.getExecutionMachine());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }

  private JobEventVO createJobEvent(JobRescheduledEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.UPDATE);
    jobEvent.setState(StatusTypeVO.RESCHEDULED.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }

  private JobEventVO createJobEvent(JobScheduledEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.CREATE);
    jobEvent.setState(StatusTypeVO.SCHEDULED.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setProjectId(e.getProjectId());
    jobEvent.setGroupId(e.getGroupId());
    if (e.getSpecificationType().toString().equals("ALGORITHM")) {
      Algorithm algo = (Algorithm) e.getSpecification();
      jobEvent.addAlgorithm(
          new AlgorithmSpecificationVO(algo.getAlgorithmName(), algo.getAlgorithmVersion()));
    } else {
      List<Algorithm> algorithms = ((Flow) e.getSpecification()).getAlgorithms();
      algorithms.stream().map(
          algo -> new AlgorithmSpecificationVO(algo.getAlgorithmName(), algo.getAlgorithmVersion()))
          .forEach(jobEvent::addAlgorithm);
    }
    jobEvent.setDescription(e.getDescription());
    jobEvent.setPriority(e.getPriority());
    jobEvent.setSelectedMachines(e.getSelectedMachines());
    jobEvent.setSubmissionTime(e.getSubmissionTime());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }

  private JobEventVO createJobEvent(JobFinishedEvent e) {
    JobEventVO jobEvent = new JobEventVO();
    jobEvent.setType(JobEventVO.JobEventType.UPDATE);
    jobEvent.setState(StatusTypeVO.FINISHED.toString());
    jobEvent.setJobId(e.getJobId());
    jobEvent.setExitCode(e.getExitCode());
    jobEvent.setExitStatus(e.getExitStatus().toString());
    jobEvent.setExitCode(e.getExitCode());
    jobEvent.setCpuTime(e.getCpuTime());
    jobEvent.setWallclockTime(e.getWallclockTime());
    jobEvent.setTimestampType(e.getTimestamp());
    return jobEvent;
  }
}
