package br.pucrio.tecgraf.soma.job;

import br.pucrio.tecgraf.soma.job.event.JobHistoryEventFactory;
import br.pucrio.tecgraf.soma.job.event.IJobHistoryEventFactory;
import csbase.server.plugin.service.IServiceManager;
import csbase.server.plugin.service.commandpersistenceservice.ICommandInfo;
import csbase.server.plugin.service.commandpersistenceservice.ICommandStatusListener;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import io.confluent.kafka.serializers.KafkaAvroSerializer;
import io.confluent.kafka.serializers.KafkaAvroSerializerConfig;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.serialization.StringSerializer;

import java.util.Properties;

import static org.apache.kafka.clients.producer.ProducerConfig.*;

public class SomaCommandStatusListener implements ICommandStatusListener {

	public static final String KAFKA_SCHEMA_REGISTRY_URL = "schema_registry_url";
	public static final String KAFKA_SERVER_ADDRESS = "kafka_server_address";
	public static final String KAFKA_TOPIC = "job-events";

	private String kafkaServer;
	private String schemaRegistry;
	private Properties pluginProperties;
	private CommandPersistenceService persistenceService;
	private IJobHistoryEventFactory eventFactory;
	private IServiceManager serviceManager;
	private Producer<String, JobHistoryEvent> producer;
	/** Pool de threads para tratar as notificações do servidor. */
	protected ExecutorService notifierExecutor;

	public SomaCommandStatusListener(IServiceManager serviceManager) {
		this.serviceManager = serviceManager;
		// Criado como executor de uma única thread para que os eventos sejam tratados na ordem que chegam
		this.notifierExecutor = Executors.newSingleThreadExecutor();
	}

	@Override
	public void statusChanged(ICommandInfo commandInfo) {
		// Evita problema de classloading das classes do kafka quando carregado como plug-in
		Thread.currentThread().setContextClassLoader(null);

    notifierExecutor.execute(
        () -> {
					JobHistoryEvent jobHistoryEvent = createJobHistoryEvent(commandInfo);
          if (jobHistoryEvent != null) {
            // Usando o id do comando como chave para que todas os eventos do comando caiam na mesma
            // partição
            getProducer().send(new ProducerRecord<>(
                        KAFKA_TOPIC, commandInfo.getCommandId(), jobHistoryEvent));
          }
        });
	}

  @Override
  public void setProperties(Properties properties) {
    pluginProperties = properties;
    kafkaServer = pluginProperties.getProperty(KAFKA_SERVER_ADDRESS);
    schemaRegistry = pluginProperties.getProperty(KAFKA_SCHEMA_REGISTRY_URL);
  }

  private JobHistoryEvent createJobHistoryEvent(ICommandInfo commandInfo) {
    switch (commandInfo.getStatus()) {
      case UPLOADING:
        {
					JobStageInEvent event = getEventFactory().buildStageInEvent(commandInfo);
          return new JobHistoryEvent(event.getClass().getSimpleName(), event);
        }
      case DOWNLOADING:
        {
					JobStageOutEvent event = getEventFactory().buildStageOutEvent(commandInfo);
          return new JobHistoryEvent(event.getClass().getSimpleName(), event);
        }
      case EXECUTING:
        {
					JobExecutingEvent event = getEventFactory().buildExecutingEvent(commandInfo);
          return new JobHistoryEvent(event.getClass().getSimpleName(), event);
        }
      case SCHEDULED:
        {
					JobScheduledEvent event = getEventFactory().buildScheduledEvent(commandInfo);
          return new JobHistoryEvent(event.getClass().getSimpleName(), event);
        }
			case RESCHEDULED:
			{
				JobRescheduledEvent event = getEventFactory().buildRescheduledEvent(commandInfo);
				return new JobHistoryEvent(event.getClass().getSimpleName(), event);
			}
      case FINISHED:
        {
					JobFinishedEvent event = getEventFactory().buildFinishedEvent(commandInfo);
          return new JobHistoryEvent(event.getClass().getSimpleName(), event);
        }
      case INIT:
	// Estado não deve produzir nenhum evento
      	return null;
      default:
        throw new IllegalArgumentException("Unsupported ICommandInfo status");
    }
  }

	protected Producer<String, JobHistoryEvent> getProducer() {
		if(producer == null) {
			Properties properties = new Properties();
	        properties.setProperty(BOOTSTRAP_SERVERS_CONFIG, kafkaServer);
	        properties.setProperty(KafkaAvroSerializerConfig.SCHEMA_REGISTRY_URL_CONFIG, schemaRegistry);
	        properties.setProperty(KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
	        properties.setProperty(VALUE_SERIALIZER_CLASS_CONFIG, KafkaAvroSerializer.class.getName());
	        properties.setProperty(ACKS_CONFIG, "1");
	        properties.setProperty(RETRIES_CONFIG, "3");
	        properties.setProperty(LINGER_MS_CONFIG, "1");
			producer = new KafkaProducer<>(properties);
		}
		return producer;
	}
	
	protected CommandPersistenceService getPersistenceService() {
		if(persistenceService == null) {
			persistenceService = (CommandPersistenceService) serviceManager.getService(CommandPersistenceService.SERVICE_NAME);
		}
		return persistenceService;
	}
	
	protected IJobHistoryEventFactory getEventFactory(){
		if(eventFactory == null) {
			eventFactory = new JobHistoryEventFactory(getPersistenceService());
		}
		return eventFactory;
	}

}
