package br.pucrio.tecgraf.soma.job.application.appservice;

import br.pucrio.tecgraf.soma.job.api.NotFoundException;
import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.application.configuration.Constants.Config;
import br.pucrio.tecgraf.soma.job.application.service.JobService;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobView;
import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

@Service
public class JobAppService {

  @Autowired JobService service;
  @Autowired private ServiceConfiguration config;

  @Transactional
  public void markJobAsDeleted(String jobId) {
    service.markJobAsDeleted(jobId);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    service.markJobsAsDeleted(jobIds);
  }

  @Transactional
  public Tuple<List<Job>, Integer> findJobs(
      String rsqlQuery, Integer offset, Integer limit, Boolean ascending, String sortAttribute) {
    List<Job> jobs = service.findJobs(rsqlQuery, offset, limit, ascending, sortAttribute);
    int total = (int) service.count(rsqlQuery);
    return new Tuple<>(jobs, total);
  }

  @Transactional
  public Tuple<List<JobView>, Integer> findJobsFromView(
    String rsqlQuery, Integer offset, Integer limit, Boolean ascending, String sortAttribute) {
    List<JobView> jobs = service.findJobsFromView(rsqlQuery, offset, limit, ascending, sortAttribute);
    int total = (int) service.countFromView(rsqlQuery);
    return new Tuple<>(jobs, total);
  }

  /**
   * Busca os projetos que o usuário tem acesso via API REST do serviço de projetos do CSGrid.
   * @param locale o locale.
   * @param accessToken o token para autenticação da chamada à API REST.
   * @return a lista de ids dos projetos que o usuário tem acesso.
   */
  public List<String> getUserProjects(String locale, String accessToken) throws NotFoundException {
    String csgridBaseURL = config.getValue(Config.CSGRID_BASE_URL.option.getLongName());
    Client client = ClientBuilder.newClient();
    WebTarget target = client.target(csgridBaseURL);

    Builder requestBuilder = target.path("projects").path("permissions").queryParam("locale", locale).request(
        MediaType.APPLICATION_JSON).header(HttpHeaders.AUTHORIZATION, accessToken);

    Response response = requestBuilder.get();
    if (response.getStatus() != 200) {
      throw new NotFoundException(response.getStatus(), "Error while trying to retrieve user projects: " +  response.getEntity());
    }
    return response.readEntity(new GenericType<List<String>>() {});
  }

  public JobPagination buildPaginationInfo(Integer total, Integer offset, Integer limit) {
    JobPagination pagination = new JobPagination();
    pagination.total(total);
    pagination.limit(limit);
    if (offset == null) {
      pagination.offset(0);
    } else {
      int newOffset;
      if (offset >= total) {
        if ( total == 0) {
          newOffset = 0;
        } else if (total % limit == 0) {
          newOffset = total - limit;
        } else {
          newOffset = total - (total % limit);
        }
      } else {
        newOffset = offset;
      }
      pagination.offset(newOffset);
    }
    return pagination;
  }
}
