package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import br.pucrio.tecgraf.soma.job.api.JobsviewApi;
import br.pucrio.tecgraf.soma.job.api.NotFoundException;
import br.pucrio.tecgraf.soma.job.api.model.Algorithm;
import br.pucrio.tecgraf.soma.job.api.model.AlgorithmParameter;
import br.pucrio.tecgraf.soma.job.api.model.Job;
import br.pucrio.tecgraf.soma.job.api.model.JobData;
import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.api.model.JobView;
import br.pucrio.tecgraf.soma.job.api.model.JobViewData;
import br.pucrio.tecgraf.soma.job.api.model.JobViewResponse;
import br.pucrio.tecgraf.soma.job.api.model.StatusChangeHistory;
import br.pucrio.tecgraf.soma.job.api.model.StatusType;
import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;


@Component
public class JobViewController extends JobsviewApi {

  @Autowired private JobAppService jobAppService;
  @Autowired private HttpServletRequest request;

  public JobViewController() {
    super(null);
  }

  @Override
  public Response jobsviewGet(
    String q,
    Integer offset,
    Integer limit,
    Boolean asc,
    String attr,
    Boolean showParam,
    String locale,
    SecurityContext securityContext)
  throws NotFoundException {
    if (limit != null && limit <= 0) {
      return Response.status(HttpStatus.BAD_REQUEST.value(), "Limit must be a positive number")
                     .build();
    }
    if (offset != null && offset < 0) {
      return Response.status(
        HttpStatus.BAD_REQUEST.value(), "Offset must be a positive number or zero")
                     .build();
    }
    if (offset != null && limit == null) {
      return Response.status(HttpStatus.BAD_REQUEST.value(), "Limit is mandatory using offset")
                     .build();
    }

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    List<String> projectIds = jobAppService.getUserProjects(locale, accessToken);
    if(projectIds.isEmpty()) {
      throw new NotFoundException(HttpStatus.NOT_FOUND.value(), "User has access to no projects");
    }

    StringBuilder projectsQuery = new StringBuilder("projectId=in=(");
    for (String projectId : projectIds) {
      projectsQuery.append(projectId);
      projectsQuery.append(",");
    }
    projectsQuery.deleteCharAt(projectsQuery.lastIndexOf(","));
    projectsQuery.append(")");

    String baseQuery = "isDeleted==false;" + projectsQuery; // por padrao só retornaremos os jobs não deletados
    if (q == null || q.length() == 0 ) { // caso de query vazia
      q = baseQuery;
    } else if( q.contains("isDeleted") == false ) { // se a busca não for pelo atributo isDeleted
      q = String.format("(%s);(%s)",q, baseQuery);
    }

    Tuple<List<br.pucrio.tecgraf.soma.job.domain.model.JobView>, Integer> tuple = jobAppService.findJobsFromView(q,
      offset, limit, asc, attr);

    JobPagination pagination = jobAppService.buildPaginationInfo(tuple.getSecond(), offset, limit);

    if (offset != pagination.getOffset()) {//offset foi alterado temos que refezer a query
      tuple = jobAppService.findJobsFromView(q, pagination.getOffset(), limit, asc, attr);
    }

    // Retorna
    return responseJobs(tuple, pagination, showParam);
  }

  private Response responseJobs(Tuple<List<br.pucrio.tecgraf.soma.job.domain.model.JobView>, Integer> tuple,
    JobPagination pagination, boolean convertToJob) {

    if (convertToJob) {
      JobData data = new JobData();
      data.processingDate(System.currentTimeMillis());
      List<Job> convertedJobList = convertToJobList(DomainMapper.convert(tuple.getFirst(), JobView[].class));
      data.jobs(convertedJobList);

      JobResponse response = new JobResponse();
      response.data(data);
      response.pagination(pagination);
      return Response.ok().entity(response).build();
    }

    JobViewData data = new JobViewData();
    data.processingDate(System.currentTimeMillis());
    data.jobs(Arrays.asList(DomainMapper.convert(tuple.getFirst(), JobView[].class)));
    JobViewResponse response = new JobViewResponse();
    response.data(data);
    response.pagination(pagination);

    return Response.ok().entity(response).build();
  }

  private List<Job> convertToJobList(JobView[] jobs) {
    Map<String, Job> jobsMap = new LinkedHashMap();

    Map<String, Map<String, Boolean>> jobStatusMap = new HashMap<>();
    Map<String, Map<String, Algorithm>> jobAlgoMap = new HashMap<>();
    Map<String, Map<String, AlgorithmParameter>> jobAlgoParamMap = new HashMap<>();

    for (JobView jobView: jobs) {
      String jobId = jobView.getJobId();

      if (!jobsMap.containsKey(jobId)) {
        Job newJob = createJob(jobView);

        jobStatusMap.put(jobId, new HashMap<>());
        jobAlgoMap.put(jobId, new HashMap<>());
        jobAlgoParamMap.put(jobId, new HashMap<>());

        jobsMap.put(jobId, newJob);
      }

      if (! jobView.getAutomaticallyMachineSelection()) {
        String selectedMachine = jobView.getSelectedMachine();
        Job job = jobsMap.get(jobId);
        List<String> submissionMachines = job.getSubmissionMachines();
        if (submissionMachines != null) {
          if (!submissionMachines.contains(selectedMachine)) {
            job.addSubmissionMachinesItem(selectedMachine);
          }
        } else {
          job.addSubmissionMachinesItem(selectedMachine);
        }
      }

      // StatusHistory
      Map<String, Boolean> statusHistoryMap = jobStatusMap.get(jobId);
      StatusType statusType = jobView.getJobStatus();
      String statusStr = statusType.toString();
      if (!statusHistoryMap.containsKey(statusStr)) {
        StatusChangeHistory sch = new StatusChangeHistory();
        sch.setStatus(statusType);
        sch.setTimestamp(jobView.getJobStatusTimestamp());

        statusHistoryMap.put(statusStr, true);

        jobsMap.get(jobId).addStatusHistoryItem(sch);
      }

      // AlgorithmParameter
      Map<String, AlgorithmParameter> paramMap = jobAlgoParamMap.get(jobId);
      String paramId = jobView.getParameterId();
      if (!paramMap.containsKey(paramId)) {
        AlgorithmParameter algoParameter = new AlgorithmParameter();
        algoParameter.setId(jobView.getJobAlgoParamId());
        algoParameter.setParameterId(jobView.getParameterId());
        algoParameter.setLabel(jobView.getParamLabel());
        algoParameter.setType(jobView.getParamType());

        paramMap.put(paramId, algoParameter);
      }
      AlgorithmParameter algoParameter = paramMap.get(paramId);
      // O valor é acrescid ao parâmetro do algoritmo
      List<String> parameterValues = algoParameter.getValue();
      if (parameterValues == null || !parameterValues.contains(jobView.getParamValue())) {
        algoParameter.addValueItem(jobView.getParamValue());
      }

      // Algorithm
      Map<String, Algorithm> algorithmMap = jobAlgoMap.get(jobId);
      String algorithmId = jobView.getAlgorithmId();
      Long jaId = jobView.getJaId();
      String keyJaId = "";
      if (jaId != null) {
        keyJaId = Long.toString(jaId);
      }
      if (!algorithmMap.containsKey(keyJaId)) {
        Algorithm algorithm = new Algorithm();

        algorithm.setAlgorithmId(algorithmId);
        algorithm.setAlgorithmVersion(jobView.getAlgorithmVersion());
        algorithm.setAlgorithmName(jobView.getAlgorithmName());

        algorithmMap.put(keyJaId, algorithm);

        jobsMap.get(jobId).addAlgorithmsItem(algorithm);
      }
      // Busca por outros valores do parâmetro
      Algorithm algorithm = algorithmMap.get(keyJaId);
      List<AlgorithmParameter> parameters = algorithm.getParameters();
      if (parameters == null || parameters.indexOf(algoParameter) == -1) {
        algorithm.addParametersItem(algoParameter);
      }
    }

    return new LinkedList<>(jobsMap.values());
  }

  private Job createJob(JobView jobView) {

    Job newJob = new Job();
    newJob.setJobId(jobView.getJobId());
    newJob.groupId(jobView.getGroupId());
    newJob.setProjectId(jobView.getProjectId());
    newJob.jobOwner(jobView.getJobOwner());
    newJob.setAutomaticallyMachineSelection(jobView.getAutomaticallyMachineSelection());

    newJob.setNumberOfProcesses(jobView.getNumberOfProcesses());
    newJob.setNumberOfProcessesByMachine(jobView.getNumberOfProcessesByMachine());
    newJob.setSubmissionTime(jobView.getSubmissionTime());
    newJob.setDescription(jobView.getDescription());
    newJob.setPriority(jobView.getPriority());
    newJob.setMultipleExecution(jobView.getMultipleExecution());
    newJob.setJobType(jobView.getJobType());
    newJob.setNumberOfAttempts(jobView.getNumberOfRetries());
    newJob.setExecutionMachine(jobView.getExecutionMachine());
    newJob.setEndTime(jobView.getEndTime());
    newJob.setExitCode(jobView.getExitCode());
    newJob.setGuiltyNodeId(jobView.getGuiltyNodeId());
    newJob.setExitStatus(jobView.getExitStatus());
    newJob.setCpuTime(jobView.getCpuTime());
    newJob.setWallclockTime(jobView.getWallclockTime());
    newJob.setRamMemory(jobView.getRamMemory());

    newJob.setFlowId(jobView.getFlowId());
    newJob.setFlowVersion(jobView.getFlowVersion());
    newJob.setFlowName(jobView.getFlowName());
    newJob.setLastModifiedTime(jobView.getLastModifiedTime());

    return newJob;
  }
}
