package br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository;

import br.pucrio.tecgraf.soma.job.domain.model.JobView;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.impl.JPARepository;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Order;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import java.util.List;

@Transactional
@Repository
public class JobViewRepository extends JPARepository<JobView> {

  @PersistenceContext private EntityManager _entityManager;

  @Override
  public Class<JobView> getType() {
    return JobView.class;
  }

  @Override
  public EntityManager getEntityManager() {
    return _entityManager;
  }

  public void setEntityManager(EntityManager entityManager) {
    this._entityManager = entityManager;
  }

  @Override
  public List<JobView> find(JPASpecification<JobView> specification, Integer maxResult, Integer offset, Sort... sorts) {
    CriteriaBuilder criteriaBuilder = getEntityManager().getCriteriaBuilder();

    // Inner Query
    CriteriaQuery<JobView> innerQueryCriteria = criteriaBuilder.createQuery(JobView.class);
    Root<JobView> fromIn = innerQueryCriteria.from(JobView.class);
    Predicate innerPredicate = specification.toPredicate(fromIn, criteriaBuilder);

    innerQueryCriteria.select(fromIn.get("jobId"))
                      .where(innerPredicate)
                      .groupBy(fromIn.get("jobId"));
                      
    // Como é um subselect, precisamos ordenar para podermos usar o "limit" corretamente.
    for (Sort sort : sorts) {
      if (sort.isAscending()) {
        Expression<Number> orderAttr = criteriaBuilder.min(fromIn.get(sort.getAttribute()));
        innerQueryCriteria.orderBy(criteriaBuilder.asc(orderAttr));
      } else {
        Expression<Number> orderAttr = criteriaBuilder.max(fromIn.get(sort.getAttribute()));
        innerQueryCriteria.orderBy(criteriaBuilder.desc(orderAttr));
      }
    }

    TypedQuery<JobView> innerQuery = getEntityManager().createQuery(innerQueryCriteria);
    if (maxResult != null) {
      if (maxResult <= 0) {
        throw new IllegalArgumentException();
      }
      innerQuery.setMaxResults(maxResult);
    }
    if (offset != null) {
      if (offset < 0) {
        throw new IllegalArgumentException();
      }
      innerQuery.setFirstResult(offset);
    }
    List<JobView> list = innerQuery.getResultList();

    if (list.isEmpty()) {
      return list;
    }

    // Outer Query
    CriteriaQuery<JobView> outerQueryCriteria = criteriaBuilder.createQuery(JobView.class);
    Root<JobView> fromOut = outerQueryCriteria.from(JobView.class);

    outerQueryCriteria.select(fromOut)
                 .where(
                   criteriaBuilder.in(fromOut.get("jobId")).value(list));
 
    // Os status de um job estarão ordenados sempre em ordem crescente de timestamp.
    Order orderByJobStatusTimeAsc = criteriaBuilder.asc(fromOut.get("jobStatusTimestamp"));
    // Os algoritmos de um job estarão ordenados sempre em ordem crescente de id do algoritmo,
    // para não variar a ordem dos algoritmos de um fluxo.
    Order orderByJobAlgorithmIdAsc = criteriaBuilder.asc(fromOut.get("jaId"));
    // job_status_history_id
    Order orderByJobStatusHistoryIdAsc = criteriaBuilder.asc(fromOut.get("jobStatushistoryId"));
    // Sorting
    for (Sort sort : sorts) {
      outerQueryCriteria.orderBy(sort.isAscending()
                                ? criteriaBuilder.asc(fromOut.get(sort.getAttribute()))
                                : criteriaBuilder.desc(fromOut.get(sort.getAttribute())),
                                 orderByJobStatusHistoryIdAsc,
                                 orderByJobStatusTimeAsc,
                                 orderByJobAlgorithmIdAsc);
    }

    return getEntityManager().createQuery(outerQueryCriteria).getResultList();
  }

  @Override
  public long count(JPASpecification<JobView> specification) {
    CriteriaBuilder criteriaBuilder = getEntityManager().getCriteriaBuilder();

    CriteriaQuery<Long> queryCriteria = criteriaBuilder.createQuery(Long.class);
    Root<JobView> from = queryCriteria.from(JobView.class);
    Predicate innerPredicate = specification.toPredicate(from, criteriaBuilder);

    queryCriteria
      .select(criteriaBuilder.countDistinct(from.get("jobId")))
      .where(innerPredicate);

    return getEntityManager().createQuery(queryCriteria).getSingleResult().longValue();
  }
}
