package br.pucrio.tecgraf.soma.job.application.controller;

import br.pucrio.tecgraf.soma.job.api.JobsApi;
import br.pucrio.tecgraf.soma.job.api.model.*;
import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;
import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.ServiceUnavailableException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import java.util.Arrays;
import java.util.List;

@Component
public class JobController extends JobsApi {
  private static final Logger LOG = Logger.getLogger(JobController.class);
  private static final int maxResultLimit = 1000;

  @Autowired private JobAppService jobAppService;
  @Autowired private HttpServletRequest request;

  public JobController() {
    super(null);
  }

  @Override
  public Response jobsHistoryAlgorithmsGet(String q, String locale, SecurityContext securityContext)
      throws ServiceUnavailableException {
    AlgorithmResponse response = new AlgorithmResponse();
    response.setProcessingDate(System.currentTimeMillis());

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    response.setAlgorithms(
        Arrays.asList(
            DomainMapper.convert(jobAppService.findDistinctAlgorithms(q), Algorithm[].class)));

    return Response.ok().entity(response).build();
  }

  @Override
  public Response jobsHistoryJobIdCommentPut(
      String jobId, NewComment requestBody, String locale, SecurityContext securityContext) {

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    if (requestBody == null || requestBody.getNewComment() == null) {
      String errorMsg = "Missing body with newComment parameter";
      LOG.info(errorMsg);
      return Response.status(HttpStatus.BAD_REQUEST.value(), errorMsg).build();
    }

    jobAppService.editJobComment(jobId, requestBody.getNewComment());
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryDelete(
      List<String> jobIds, String locale, SecurityContext securityContext) {
    jobAppService.markJobsAsDeleted(jobIds);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryGet(
      String q,
      Integer offset,
      Integer limit,
      Boolean asc,
      String attr,
      Boolean grouped,
      String locale,
      SecurityContext securityContext) {
    if (limit == null) {
      limit = maxResultLimit;
    } else if (limit <= 0 || limit > maxResultLimit) {
      String msg = "Invalid limit value";
      LOG.info(msg);
      return Response.status(HttpStatus.BAD_REQUEST.value(), msg).build();
    }
    if (offset == null) {
      offset = 0;
    } else if (offset < 0) {
      String msg = "Offset must be a positive number or zero";
      LOG.info(msg);
      return Response.status(HttpStatus.BAD_REQUEST.value(), msg).build();
    }
    if (asc == null) {
      asc = false;
    }

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    JobData data = new JobData();
    data.processingDate(System.currentTimeMillis());
    JobPagination pagination = new JobPagination();
    pagination.setLimit(limit);
    JobResponse response = new JobResponse();
    response.data(data);
    response.pagination(pagination);

    if (grouped) {
      jobAppService.findGroupedJobs(q, limit, offset, asc, attr, response);
    } else {
      jobAppService.findJobs(q, limit, offset, asc, attr, response);
    }

    return Response.ok().entity(response).build();
  }
}
