package br.pucrio.tecgraf.soma.job.application.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.github.tennaito.rsql.misc.SimpleMapper;

import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.JsonUndefined;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobAlgorithmRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobGroupValCount;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl.RSQLSpecFactory;

@Service
public class JobService {
  private static final Logger LOG = Logger.getLogger(JobService.class);
  private static final SimpleMapper mapper = new SimpleMapper(2);

  static {
    Map<String, String> jobAttribAliases = new HashMap<>(10);
    jobAttribAliases.put("jobStatus", "statusHistory.status");
    jobAttribAliases.put("jobStatusTimestamp", "statusHistory.timestamp");
    jobAttribAliases.put("algorithmId", "algorithms.algorithmId");
    jobAttribAliases.put("algorithmVersion", "algorithms.algorithmVersion");
    jobAttribAliases.put("algorithmName", "algorithms.algorithmName");
    jobAttribAliases.put("flowNodeId", "algorithms.flowNodeId");
    jobAttribAliases.put("parameterId", "algorithms.parameters.parameterId");
    jobAttribAliases.put("paramLabel", "algorithms.parameters.label");
    jobAttribAliases.put("paramType", "algorithms.parameters.type");
    jobAttribAliases.put("paramValue", "algorithms.parameters.value");
    mapper.addMapping(Job.class, jobAttribAliases);

    Map<String, String> jobJobAlgorithmAttribAliases = new HashMap<>(3);
    jobJobAlgorithmAttribAliases.put("jobStatus", "job.statusHistory.status");
    jobJobAlgorithmAttribAliases.put("exitStatus", "job.exitStatus");
    jobJobAlgorithmAttribAliases.put("groupId", "job.groupId");
    mapper.addMapping(JobAlgorithm.class, jobJobAlgorithmAttribAliases);
  }

  private final RSQLSpecFactory<Job> jobRsqlBuilder;
  private final RSQLSpecFactory<JobAlgorithm> jobAlgoRsqlBuilder;

  @Autowired private JobRepository jobRepository;
  @Autowired private JobAlgorithmRepository jobAlgorithmRepository;

  JobService(JobRepository jobRepository, JobAlgorithmRepository jobAlgorithmRepository) {
    this.jobRepository = jobRepository;
    this.jobAlgorithmRepository = jobAlgorithmRepository;

    this.jobRsqlBuilder = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    this.jobAlgoRsqlBuilder =
        new RSQLSpecFactory<>(jobAlgorithmRepository.getEntityManager(), mapper);
  }

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    LOG.info(String.format("Querying distinct algorithms using the query: %s", rsqlQuery));

    JPASpecification<JobAlgorithm> query = jobAlgoRsqlBuilder.create(rsqlQuery);
    return jobAlgorithmRepository.findDistinct(query);
  }

  @Transactional
  public Job findJobByStringId(String jobId) {
    return this.jobRepository.first(new JobByIdSpecification(jobId));
  }

  @Transactional
  public void editJobComment(String jobId, String newComment) {
    Job job = this.findJobByStringId(jobId);
    job.setDescription(newComment);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    Job job = this.jobRepository.first(new JobByIdSpecification(jobId));
    job.setDeleted(true);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    JobsInListSpecification spec = new JobsInListSpecification(jobIds);
    List<Job> jobs = this.jobRepository.find(spec);
    for (Job job : jobs) {
      job.setDeleted(true);
      this.jobRepository.update(job);
    }
  }

  private int adjustOffset(int offset, int limit, int total) {
    // ajusta para última página, caso esteja além dela
    if (offset > 0) {
      if (offset >= total) {
        if (total % limit == 0) {
          offset = total - limit;
        } else {
          offset = total - (total % limit);
        }
      }
    }
    return offset;
  }

  private Sort[] toSortArray(String sortAttribute, boolean ascending) {
    if (sortAttribute == null) {
      return new Sort[0];
    }
    return new Sort[] {new Sort(sortAttribute, ascending)};
  }

  @Transactional
  public void findJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    LOG.info(String.format("Querying jobs using the query: %s", rsqlQuery));

    JobPagination pagination = response.getPagination();
    JPASpecification<Job> query = jobRsqlBuilder.create(rsqlQuery);
    int total = Math.toIntExact(jobRepository.count(query));
    pagination.setTotal(total);
    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    } else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);
      List<Long> jobIds = jobRepository.findJobIds(query, limit, offset, sorting);
      List<Job> domainJobList = jobRepository.getJobs(jobIds, sorting);
      pagination.setOffset(offset);
      response
          .getData()
          .setJobs(convertDomainJobListToApiJobList(domainJobList));
    }
  }

  @Transactional
  public void findGroupedJobs(
      String rsqlQuery,
      int limit,
      int offset,
      boolean ascending,
      String sortAttribute,
      JobResponse response) {
    LOG.info(String.format("Querying grouped jobs using the query: %s", rsqlQuery));

    JobPagination pagination = response.getPagination();
    JPASpecification<Job> query = jobRsqlBuilder.create(rsqlQuery);
    int total = Math.toIntExact(jobRepository.countGroups(query));
    pagination.setTotal(total);
    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    } else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);
      List<JobGroupValCount> groups = jobRepository.findGroupedJobs(query, limit, offset, sorting);
      List<String> groupIds = new ArrayList<>(groups.size());
      Map<Long, Integer> jobId2Pos = new HashMap<>(groups.size());
      for (int pos = 0; pos < groups.size(); pos++) {
        JobGroupValCount group = groups.get(pos);
        jobId2Pos.put(group.firstJobId, pos);
        groupIds.add(group.groupId);
      }
      groupIds = jobRepository.filterSingletons(groupIds);
      List<Job> jobList = jobRepository.getJobs(jobId2Pos.keySet());
      br.pucrio.tecgraf.soma.job.api.model.Job[] apiJobs =
          new br.pucrio.tecgraf.soma.job.api.model.Job[groups.size()];
      for (Job job : jobList) {
        br.pucrio.tecgraf.soma.job.api.model.Job apiJob = this.convertDomainJobToApiJob(job);
        int i = jobId2Pos.get(job.getId());
        apiJobs[i] = apiJob;

        if (groupIds.contains(job.getGroupId())) {
          JobGroupValCount group = groups.get(i);
          apiJob.setIsGroup(true);
          apiJob.setJobId(JsonUndefined.STRING);
          apiJob.setEndTime(JsonUndefined.STRING);
          apiJob.setCpuTime(JsonUndefined.DOUBLE);
          apiJob.setWallclockTime(JsonUndefined.INTEGER);
          apiJob.setRamMemory(JsonUndefined.DOUBLE);
          apiJob.setAlgorithms(JsonUndefined.typedList());
          apiJob.setStatusHistory(JsonUndefined.typedList());

          apiJob.setSubmissionTime(group.submissionTime.toString());
          apiJob.setLastModifiedTime(group.lastModifiedTime.toString());

          // clear heterogeneous values.
          if (group.projectId > 1) apiJob.setProjectId(JsonUndefined.STRING);
          if (group.jobOwner > 1) apiJob.setJobOwner(JsonUndefined.STRING);
          if (group.automaticallyMachineSelection > 1)
            apiJob.setAutomaticallyMachineSelection(JsonUndefined.BOOLEAN);
          if (group.numberOfProcesses > 1) apiJob.setNumberOfProcesses(JsonUndefined.INTEGER);
          if (group.numberOfProcessesByMachine > 1)
            apiJob.setNumberOfProcessesByMachine(JsonUndefined.INTEGER);
          if (group.description > 1) apiJob.setDescription(JsonUndefined.STRING);
          if (group.priority > 1) apiJob.setPriority(JsonUndefined.INTEGER);
          if (group.multipleExecution > 1) apiJob.setMultipleExecution(JsonUndefined.BOOLEAN);
          if (group.jobType > 1) apiJob.setJobType(null);
          if (group.executionMachine > 1) apiJob.setExecutionMachine(JsonUndefined.STRING);
          if (group.exitCode > 1) apiJob.setExitCode(JsonUndefined.INTEGER);
          if (group.guiltyNodeId > 1) apiJob.setGuiltyNodeId(JsonUndefined.STRING);
          if (group.exitStatus > 1) apiJob.setExitStatus(null);
          if (group.flowId > 1) apiJob.setFlowId(JsonUndefined.STRING);
          if (group.flowVersion > 1) apiJob.setFlowVersion(JsonUndefined.STRING);
          if (group.flowName > 1) apiJob.setFlowName(JsonUndefined.STRING);
        }
      }
      pagination.setOffset(offset);
      response.getData().setJobs(Arrays.asList(apiJobs));
    }
  }

  private List<br.pucrio.tecgraf.soma.job.api.model.Job> convertDomainJobListToApiJobList(List<Job> domainJobList) {
    return Arrays.asList(
      DomainMapper.convert(domainJobList, br.pucrio.tecgraf.soma.job.api.model.Job[].class));
  }

  private br.pucrio.tecgraf.soma.job.api.model.Job convertDomainJobToApiJob(Job domainJob) {
    return DomainMapper.convert(domainJob, br.pucrio.tecgraf.soma.job.api.model.Job.class);
  }
}
