package br.pucrio.tecgraf.soma.job.application.appservice;

import java.time.LocalDateTime;
import java.time.ZoneOffset;

import javax.ws.rs.ForbiddenException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.HttpClientErrorException;

import br.pucrio.tecgraf.soma.job.api.model.ReplicaJob;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaJobResponse;
import br.pucrio.tecgraf.soma.job.application.service.MultiflowService;
import br.pucrio.tecgraf.soma.job.application.service.ProjectService;
import br.pucrio.tecgraf.soma.job.application.service.ReplicaService;
import br.pucrio.tecgraf.soma.job.domain.model.Replica;

@Service
public class ReplicaAppService {

  @Autowired private ProjectService projectService;
  @Autowired private MultiflowService multiflowService;
  @Autowired private ReplicaService replicaService;

   /**
   * Persiste os dados de uma execução de uma réplica.
   *
   * @param replicaJobData dados da execução de uma réplica.
   * @return o id da réplica
   */
  @Transactional
  public ReplicaJobResponse createReplicaJob(ReplicaJob replicaJobData) {
    br.pucrio.tecgraf.soma.job.domain.model.Multiflow multiflow = multiflowService.findMultiflowById(replicaJobData.getMultiflowId());

    if (!projectService.hasPermission(multiflow.getProjectId())) {
      throw new ForbiddenException("User has no permission to add a Replica to this project");
    }

    br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJobFound = replicaService.findReplicaJobBy(replicaJobData.getJobId());
    if (replicaJobFound != null) {
      throw new HttpClientErrorException(HttpStatus.CONFLICT, "Job ID " + replicaJobData.getJobId() + " exists in line number " + replicaJobFound.getReplica().getLineNumber());
    }

    Replica replica = replicaService.findReplicaBy(multiflow.getId(), replicaJobData.getLineNumber().intValue());
    if(replica == null) {
      replica = new Replica();
      replica.setMultiflow(multiflow);
      replica.setLineNumber(replicaJobData.getLineNumber());
      replicaService.createReplica(replica);
    }

    br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJob = new br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob();
    // TODO: eliminar a possibilidade de números de versão repetidos para uma mesma réplica (WEBSNTS-19170)
    replicaJob.setVersion(replicaService.getNextReplicaJobVersion(replica));
    replicaJob.setJobStringId(replicaJobData.getJobId());
    replicaJob.setSubmissionTime(LocalDateTime.now(ZoneOffset.UTC));
    replicaJob.setReplicaDependencyRaw(replicaJobData.getReplicaDependencyRaw());
    replica.addReplicaJob(replicaJob);
    replicaService.updateReplica(replica);

    return convertToBasicRESTModel(replicaService.findReplicaJobBy(replicaJob.getJobStringId()));
  }

  private static ReplicaJobResponse convertToBasicRESTModel(br.pucrio.tecgraf.soma.job.domain.model.ReplicaJob replicaJobDbModel) {
    ReplicaJobResponse restModel = new ReplicaJobResponse();
    Replica replica = replicaJobDbModel.getReplica();
    restModel.setLineNumber(replica.getLineNumber());
    restModel.setMultiflowId(replica.getMultiflow().getId());
    restModel.setMultiflowName(replica.getMultiflow().getName());
    restModel.setJobId(replicaJobDbModel.getJobStringId());
    restModel.setVersion(replicaJobDbModel.getVersion());
    restModel.replicaDependencyRaw(replicaJobDbModel.getReplicaDependencyRaw());
    return restModel;
  }
}
