package br.pucrio.tecgraf.soma.job.application.controller;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import br.pucrio.tecgraf.soma.job.api.JobsApi;
import br.pucrio.tecgraf.soma.job.api.NotFoundException;
import br.pucrio.tecgraf.soma.job.api.model.Algorithm;
import br.pucrio.tecgraf.soma.job.api.model.AlgorithmParameter;
import br.pucrio.tecgraf.soma.job.api.model.AlgorithmResponse;
import br.pucrio.tecgraf.soma.job.api.model.ExitStatus;
import br.pucrio.tecgraf.soma.job.api.model.Job;
import br.pucrio.tecgraf.soma.job.api.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.api.model.JobData;
import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.api.model.JobType;
import br.pucrio.tecgraf.soma.job.api.model.StatusChangeHistory;
import br.pucrio.tecgraf.soma.job.api.model.StatusType;
import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;
import br.pucrio.tecgraf.soma.job.domain.model.JobView;

@Component
public class JobController extends JobsApi {

  @Autowired private JobAppService jobAppService;
  @Autowired private HttpServletRequest request;
  private static final Integer maximumLimit = 1000;
  private static final Integer invalidLimit = -1;

  public JobController() {
    super(null);
  }

  @Override
  public Response jobsHistoryAlgorithmsGet(String q, String locale, SecurityContext securityContext)
      throws NotFoundException {

    AlgorithmResponse response = new AlgorithmResponse();
    response.setProcessingDate(System.currentTimeMillis());

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjects(locale, accessToken, q);

    response.setAlgorithms(
            Arrays.asList(DomainMapper.convert(jobAppService.findDistinctAlgorithms(q), Algorithm[].class)));

    return Response.ok().entity(response).build();
  }

  @Override
  public Response jobsHistoryDelete(List<String> jobIds, String locale, SecurityContext securityContext)
      throws NotFoundException {
    jobAppService.markJobsAsDeleted(jobIds);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  private Integer getLimitValue(Integer limit) {
    if (limit == null) {
      return maximumLimit;
    }
    else if (limit > 0 && limit <= maximumLimit) {
      return limit;
    }
    return invalidLimit;
  }

  @Override
  public Response jobsHistoryGet(
      String q,
      Integer offset,
      Integer limit,
      Boolean asc,
      String attr,
      Boolean showParam,
      String locale,
      SecurityContext securityContext)
      throws NotFoundException {


    // Valor obrigatório para o request
    limit = getLimitValue(limit);

    if (limit == invalidLimit) {
      return Response.status(HttpStatus.BAD_REQUEST.value(), "Invalid limit value")
          .build();
    }
    if (offset != null && offset < 0) {
      return Response.status(
          HttpStatus.BAD_REQUEST.value(), "Offset must be a positive number or zero")
          .build();
    }

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjectsAndHandleDeleted(locale, accessToken, q);

    Tuple<List<JobView>, Integer> tuple = jobAppService.findJobsFromView(q,
        offset, limit, asc, attr);

    if(tuple == null) {
      JobResponse response = new JobResponse();
      return Response.ok().entity(response).build();
    }
    JobPagination pagination = jobAppService.buildPaginationInfo(tuple.getSecond(), offset, limit);

    if (offset != pagination.getOffset()) {//offset foi alterado temos que refezer a query
      tuple = jobAppService.findJobsFromView(q, pagination.getOffset(), limit, asc, attr);
    }

    // Retorna
    return responseJobs(tuple, pagination, showParam);
  }

  public String filterUserProjects(String locale, String accessToken, String q)
      throws NotFoundException {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "job.projectId");
    if (q == null || q.length() == 0) { // caso de query vazia
      q = projectsQuery;
    } else {
      q = String.format("(%s);(%s)", q, projectsQuery);
    }
    return q;
  }

  public String filterUserProjectsAndHandleDeleted(String locale, String accessToken, String q)
      throws NotFoundException {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "projectId");
    String baseQuery = "isDeleted==false;" + projectsQuery; // por padrao só retornaremos os jobs não deletados
    if (q == null || q.length() == 0 ) { // caso de query vazia
      q = baseQuery;
    } else if( q.contains("isDeleted") == false ) { // se a busca NÃO for pelo atributo isDeleted
      // Adiciona restrição de projetos do usuário e de jobs não apagados
      q = String.format("(%s);(%s)",q, baseQuery);
    } else {
      // Adiciona apenas restrição de projetos do usuário
      q = String.format("(%s);(%s)", q, projectsQuery);
    }

    return q;
  }

  public String getUserProjectsPredicate(String locale, String accessToken, String projectColumn)
          throws NotFoundException {
    List<String> projectIds = jobAppService.getUserProjects(locale, accessToken);
    if(projectIds.isEmpty()) {
      throw new NotFoundException(HttpStatus.NOT_FOUND.value(), "User has access to no projects");
    }

    StringBuilder projectsQuery = new StringBuilder(projectColumn);
    projectsQuery.append("=in=(");
    for (String projectId : projectIds) {
      projectsQuery.append(projectId);
      projectsQuery.append(",");
    }
    projectsQuery.deleteCharAt(projectsQuery.lastIndexOf(","));
    projectsQuery.append(")");

    return projectsQuery.toString();
  }

  private Response responseJobs(Tuple<List<JobView>, Integer> tuple,
    JobPagination pagination, boolean showParam) {

    JobData data = new JobData();
    data.processingDate(System.currentTimeMillis());
    List<Job> convertedJobList = convertToJobList(DomainMapper.convert(tuple.getFirst(), JobView[].class), showParam);
    data.jobs(convertedJobList);

    JobResponse response = new JobResponse();
    response.data(data);
    response.pagination(pagination);

    return Response.ok().entity(response).build();
  }

  private List<Job> convertToJobList(JobView[] jobs, boolean showParam) {
    Map<String, Job> jobsMap = new LinkedHashMap();

    Map<String, Map<String, Boolean>> jobStatusMap = new HashMap<>();
    Map<String, Map<String, JobAlgorithm>> jobAlgoMap = new HashMap<>();
    Map<String, Map<String, AlgorithmParameter>> jobAlgoParamMap = new HashMap<>();

    for (JobView jobView: jobs) {
      String jobId = jobView.getJobId();

      if (!jobsMap.containsKey(jobId)) {
        Job newJob = createJob(jobView);

        jobStatusMap.put(jobId, new HashMap<>());
        jobAlgoMap.put(jobId, new HashMap<>());
        jobAlgoParamMap.put(jobId, new HashMap<>());

        jobsMap.put(jobId, newJob);
      }

      if (!jobView.getAutomaticallyMachineSelection()) {
        String selectedMachine = jobView.getSelectedMachine();
        Job job = jobsMap.get(jobId);
        List<String> submissionMachines = job.getSubmissionMachines();
        if (submissionMachines != null) {
          if (!submissionMachines.contains(selectedMachine)) {
            job.addSubmissionMachinesItem(selectedMachine);
          }
        }
        else {
          job.addSubmissionMachinesItem(selectedMachine);
        }
      }

      // StatusHistory
      Map<String, Boolean> statusHistoryMap = jobStatusMap.get(jobId);
      br.pucrio.tecgraf.soma.job.domain.model.StatusType statusType = jobView.getJobStatus();
      String statusStr = statusType.toString();
      if (!statusHistoryMap.containsKey(statusStr)) {
        StatusChangeHistory sch = new StatusChangeHistory();
        sch.setStatus(DomainMapper.convert(statusType, StatusType.class));
        sch.setTimestamp(jobView.getJobStatusTimestamp().toString());

        statusHistoryMap.put(statusStr, true);

        jobsMap.get(jobId).addStatusHistoryItem(sch);
      }

      // Algorithm
      Map<String, JobAlgorithm> algorithmMap = jobAlgoMap.get(jobId);
      String algorithmId = jobView.getAlgorithmId();
      Long jaId = jobView.getJaId();
      String keyJaId = "";
      if (jaId != null) {
        keyJaId = Long.toString(jaId);
      }
      if (!algorithmMap.containsKey(keyJaId)) {
        JobAlgorithm jobAlgorithm = new JobAlgorithm();

        jobAlgorithm.setAlgorithmId(algorithmId);
        jobAlgorithm.setAlgorithmVersion(jobView.getAlgorithmVersion());
        jobAlgorithm.setAlgorithmName(jobView.getAlgorithmName());
        jobAlgorithm.setFlowNodeId(jobView.getFlowNodeId());
        jobAlgorithm.setParameters(new ArrayList<>());

        algorithmMap.put(keyJaId, jobAlgorithm);

        jobsMap.get(jobId).addAlgorithmsItem(jobAlgorithm);
      }

      // AlgorithmParameter
      Map<String, AlgorithmParameter> paramMap = jobAlgoParamMap.get(jobId);
      // Jobview pode retornar algoritmos sem parâmetros
      if (jobView.getParameterId() != null && jobView.getParamValue() != null && !jobView.getParamValue().isEmpty()) {
        // o paramID precisa ser concatenado com o identificador do algoritmo para tratamento dos casos com dois ou mais algoritmos com mesmo nome e versão no fluxo.
        String paramId = jobView.getParameterId() + keyJaId;
        if (!paramMap.containsKey(paramId)) {
          AlgorithmParameter algoParameter = new AlgorithmParameter();
          algoParameter.setId(BigDecimal.valueOf(jobView.getJobAlgoParamId()));
          algoParameter.setParameterId(jobView.getParameterId());
          algoParameter.setLabel(jobView.getParamLabel());
          algoParameter.setType(jobView.getParamType());
          paramMap.put(paramId, algoParameter);
        }
        AlgorithmParameter algoParameter = paramMap.get(paramId);
        // O valor é acrescid ao parâmetro do algoritmo
        List<String> parameterValues = algoParameter.getValue();
        if (parameterValues == null || !parameterValues.contains(jobView.getParamValue())) {
          algoParameter.addValueItem(jobView.getParamValue());
        }
        if(showParam) {
          // Busca por outros valores do parâmetro
          JobAlgorithm jobAlgorithm = algorithmMap.get(keyJaId);
          List<AlgorithmParameter> parameters = jobAlgorithm.getParameters();
          if (parameters == null || parameters.indexOf(algoParameter) == -1) {
            jobAlgorithm.addParametersItem(algoParameter);
          }
        }
      }

    }

    return new LinkedList<>(jobsMap.values());
  }

  private Job createJob(JobView jobView) {

    Job newJob = new Job();
    newJob.setJobId(jobView.getJobId());
    newJob.groupId(jobView.getGroupId());
    newJob.setProjectId(jobView.getProjectId());
    newJob.jobOwner(jobView.getJobOwner());
    newJob.setAutomaticallyMachineSelection(jobView.getAutomaticallyMachineSelection());

    newJob.setNumberOfProcesses(jobView.getNumberOfProcesses());
    newJob.setNumberOfProcessesByMachine(jobView.getNumberOfProcessesByMachine());
    newJob.setSubmissionTime(jobView.getSubmissionTime().toString());
    newJob.setDescription(jobView.getDescription());
    newJob.setPriority(jobView.getPriority());
    newJob.setMultipleExecution(jobView.getMultipleExecution());
    newJob.setJobType(DomainMapper.convert(jobView.getJobType(), JobType.class));
    newJob.setNumberOfAttempts(jobView.getNumberOfRetries());
    newJob.setExecutionMachine(jobView.getExecutionMachine());
    if (jobView.getEndTime() != null) {
      newJob.setEndTime(jobView.getEndTime().toString());
    }
    newJob.setExitCode(jobView.getExitCode());
    newJob.setGuiltyNodeId(jobView.getGuiltyNodeId());
    if(jobView.getExitStatus() != null) {
      newJob.setExitStatus(DomainMapper.convert(jobView.getExitStatus(), ExitStatus.class));
    }
    newJob.setCpuTime(jobView.getCpuTime());
    newJob.setWallclockTime(jobView.getWallclockTime());
    newJob.setRamMemory(jobView.getRamMemory());

    newJob.setFlowId(jobView.getFlowId());
    newJob.setFlowVersion(jobView.getFlowVersion());
    newJob.setFlowName(jobView.getFlowName());
    if(jobView.getLastModifiedTime() != null) {
      newJob.setLastModifiedTime(jobView.getLastModifiedTime().toString());
    }

    return newJob;
  }}
