package br.pucrio.tecgraf.soma.job.application.service;

import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobView;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobAlgorithmRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobViewRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl.RSQLSpecification;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.core.Response;
import javax.ws.rs.NotFoundException;

@Service
public class JobService {

  @Autowired private JobRepository jobRepository;
  @Autowired private JobViewRepository jobViewRepository;
  @Autowired private JobAlgorithmRepository jobAlgorithmRepository;

  JobService(){
    super();
  }

  JobService(JobRepository jobRepository) {
    this.jobRepository = jobRepository;
  }

  JobService(JobRepository jobRepository, JobViewRepository jobViewRepository) {
    this.jobRepository = jobRepository;
    this.jobViewRepository = jobViewRepository;
  }

  JobService(JobRepository jobRepository, JobViewRepository jobViewRepository, JobAlgorithmRepository jobAlgorithmRepository) {
    this.jobRepository = jobRepository;
    this.jobViewRepository = jobViewRepository;
    this.jobAlgorithmRepository = jobAlgorithmRepository;
  }

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    return jobAlgorithmRepository.findDistinct(
            new RSQLSpecification<>(rsqlQuery, jobAlgorithmRepository.getEntityManager()));
  }

  @Transactional
  public void editJobComment(String jobId, String newComment, List<String> userProjects)
          throws NotFoundException, ForbiddenException {
    Job job = null;
    try {
      job = this.jobRepository.first(new JobByIdSpecification(jobId));
    }
    catch(javax.persistence.NoResultException e) {
      System.err.println("Job " + jobId + " not found!");
      e.printStackTrace();
    }

    if(job == null) {
      throw new NotFoundException("Job not found: " + jobId);
    }
    if(!userProjects.contains(job.getProjectId())) {
      throw new ForbiddenException("User has no permission to edit this job");
    }
    job.setDescription(newComment);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    Job job = this.jobRepository.first(new JobByIdSpecification(jobId));
    job.setDeleted(true);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    JobsInListSpecification spec = new JobsInListSpecification(jobIds);
    List<Job> jobs = this.jobRepository.find(spec);
    for (Job job : jobs) {
      job.setDeleted(true);
      this.jobRepository.update(job);
    }
  }

  @Transactional
  public List<JobView> findJobsFromView(
    String rsqlQuery, Integer offset, Integer limit, Boolean ascending, String sortAttribute) {
    List<Sort> sorts = new ArrayList<>();
    if (ascending != null) {
      sorts.add(new Sort(sortAttribute, ascending));
    }

    return jobViewRepository.find(
      new RSQLSpecification<>(rsqlQuery, jobViewRepository.getEntityManager()),
      limit,
      offset,
      sorts.toArray(new Sort[0]));
  }

  @Transactional
  public long countFromView(String rsqlQuery) {
    return jobViewRepository.count(
      new RSQLSpecification<>(rsqlQuery, jobViewRepository.getEntityManager()));
  }
}
