package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.ServiceUnavailableException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import br.pucrio.tecgraf.soma.job.api.JobsApi;
import br.pucrio.tecgraf.soma.job.api.NotFoundException;
import br.pucrio.tecgraf.soma.job.api.model.Algorithm;
import br.pucrio.tecgraf.soma.job.api.model.AlgorithmResponse;
import br.pucrio.tecgraf.soma.job.api.model.NewComment;
import br.pucrio.tecgraf.soma.job.api.model.Job;
import br.pucrio.tecgraf.soma.job.api.model.JobData;
import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.application.appservice.JobAppService;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;

@Component
public class JobController extends JobsApi {

  @Autowired private JobAppService jobAppService;
  @Autowired private HttpServletRequest request;
  private static final int maxResultLimit = 1000;

  public JobController() {
    super(null);
  }

  @Override
  public Response jobsHistoryAlgorithmsGet(String q, String locale, SecurityContext securityContext)
      throws NotFoundException, ServiceUnavailableException {

    AlgorithmResponse response = new AlgorithmResponse();
    response.setProcessingDate(System.currentTimeMillis());

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjects(locale, accessToken, q);

    response.setAlgorithms(
            Arrays.asList(DomainMapper.convert(jobAppService.findDistinctAlgorithms(q), Algorithm[].class)));

    return Response.ok().entity(response).build();
  }

  @Override
  public Response jobsHistoryJobIdCommentPut(String jobId, NewComment requestBody, String locale, SecurityContext securityContext)
      throws NotFoundException, ForbiddenException, ServiceUnavailableException {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    if (requestBody == null || requestBody.getNewComment() == null) {
      return Response.status(HttpStatus.BAD_REQUEST.value(), "Missing body with newComment parameter")
          .build();
    }
    List<String> userProjects = jobAppService.getUserProjects(locale, accessToken);
    jobAppService.editJobComment(jobId, requestBody.getNewComment(), userProjects);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryDelete(List<String> jobIds, String locale, SecurityContext securityContext)
      throws NotFoundException {
    jobAppService.markJobsAsDeleted(jobIds);
    return Response.status(Response.Status.NO_CONTENT).build();
  }

  @Override
  public Response jobsHistoryGet(
      String q,
      Integer offset,
      Integer limit,
      Boolean asc,
      String attr,
      Boolean showParam,
      String locale,
      SecurityContext securityContext)
      throws NotFoundException, ServiceUnavailableException {

    if (limit == null) {
      limit = maxResultLimit;
    } else if (limit <= 0 || limit > maxResultLimit) {
      return Response.status(HttpStatus.BAD_REQUEST.value(),
              "Invalid limit value").build();
    }
    if (offset == null) {
      offset = 0;
    } else if (offset < 0) {
      return Response.status(HttpStatus.BAD_REQUEST.value(),
              "Offset must be a positive number or zero").build();
    }
    if (asc == null) {
      asc = false;
    }

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if(accessToken == null || accessToken.isEmpty()) {
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }
    q = this.filterUserProjectsAndHandleDeleted(locale, accessToken, q);

    Tuple<List<br.pucrio.tecgraf.soma.job.domain.model.Job>, Integer> found =
            jobAppService.findJobs(q, offset, limit, asc, attr);
    List<Job> jobList = Arrays.asList(DomainMapper.convert(found.getFirst(), Job[].class));
    JobPagination pagination = buildPaginationInfo(found.getSecond(), offset, limit);
    return responseJobs(jobList, pagination);
  }

  public String filterUserProjects(String locale, String accessToken, String q)
      throws NotFoundException, ServiceUnavailableException {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "job.projectId");
    if (q == null || q.length() == 0) { // caso de query vazia
      q = projectsQuery;
    } else {
      q = String.format("(%s);(%s)", q, projectsQuery);
    }
    return q;
  }

  public String filterUserProjectsAndHandleDeleted(String locale, String accessToken, String q)
      throws NotFoundException, ServiceUnavailableException {

    String projectsQuery = this.getUserProjectsPredicate(locale, accessToken, "projectId");
    String baseQuery = "isDeleted==false;" + projectsQuery; // por padrao só retornaremos os jobs não deletados
    if (q == null || q.length() == 0 ) { // caso de query vazia
      q = baseQuery;
    } else if( q.contains("isDeleted") == false ) { // se a busca NÃO for pelo atributo isDeleted
      // Adiciona restrição de projetos do usuário e de jobs não apagados
      q = String.format("(%s);(%s)",q, baseQuery);
    } else {
      // Adiciona apenas restrição de projetos do usuário
      q = String.format("(%s);(%s)", q, projectsQuery);
    }

    return q;
  }

  public String getUserProjectsPredicate(String locale, String accessToken, String projectColumn)
          throws NotFoundException, ServiceUnavailableException {
    List<String> projectIds = jobAppService.getUserProjects(locale, accessToken);
    if(projectIds.isEmpty()) {
      throw new NotFoundException(HttpStatus.NOT_FOUND.value(), "User has access to no projects");
    }
    StringBuilder projectsQuery = new StringBuilder(projectColumn);
    projectsQuery.append("=in=(");
    for (String projectId : projectIds) {
      projectsQuery.append(projectId);
      projectsQuery.append(",");
    }
    projectsQuery.deleteCharAt(projectsQuery.lastIndexOf(","));
    projectsQuery.append(")");

    return projectsQuery.toString();
  }

  private JobPagination buildPaginationInfo(int total, int offset, int limit) {
    JobPagination pagination = new JobPagination();
    pagination.total(total);
    pagination.offset(offset);
    pagination.limit(limit);
    return pagination;
  }

  private Response responseJobs(List<Job> jobList, JobPagination pagination) {

    JobData data = new JobData();
    data.processingDate(System.currentTimeMillis());
    data.jobs(jobList);

    JobResponse response = new JobResponse();
    response.data(data);
    response.pagination(pagination);

    return Response.ok().entity(response).build();
  }

}