package br.pucrio.tecgraf.soma.job.application.service;

import com.github.tennaito.rsql.misc.SimpleMapper;

import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobAlgorithmRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl.RSQLSpecFactory;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;

@Service
public class JobService {

  @Autowired private JobRepository jobRepository;
  @Autowired private JobAlgorithmRepository jobAlgorithmRepository;

  private final RSQLSpecFactory<Job> jobRsqlBuilder;
  private final RSQLSpecFactory<JobAlgorithm> jobAlgoRsqlBuilder;

  private static final SimpleMapper mapper = new SimpleMapper(1);

  static {
    Map<String, String> jobAttribAliases = new HashMap<>(10);
    jobAttribAliases.put("jobStatus"         , "statusHistory.status");
    jobAttribAliases.put("jobStatusTimestamp", "statusHistory.timestamp");
    jobAttribAliases.put("algorithmId"       , "algorithms.algorithmId");
    jobAttribAliases.put("algorithmVersion"  , "algorithms.algorithmVersion");
    jobAttribAliases.put("algorithmName"     , "algorithms.algorithmName");
    jobAttribAliases.put("flowNodeId"        , "algorithms.flowNodeId");
    jobAttribAliases.put("parameterId"       , "algorithms.parameters.parameterId");
    jobAttribAliases.put("paramLabel"        , "algorithms.parameters.label");
    jobAttribAliases.put("paramType"         , "algorithms.parameters.type");
    jobAttribAliases.put("paramValue"        , "algorithms.parameters.value");
    mapper.addMapping(Job.class, jobAttribAliases);
  }

  JobService(JobRepository jobRepository, JobAlgorithmRepository jobAlgorithmRepository) {
    this.jobRepository = jobRepository;
    this.jobAlgorithmRepository = jobAlgorithmRepository;

    this.jobRsqlBuilder = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    this.jobAlgoRsqlBuilder = new RSQLSpecFactory<>(jobAlgorithmRepository.getEntityManager());
  }

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    JPASpecification<JobAlgorithm> query = jobAlgoRsqlBuilder.create(rsqlQuery);
    return jobAlgorithmRepository.findDistinct(query);
  }

  @Transactional
  public void editJobComment(String jobId, String newComment, List<String> userProjects)
          throws NotFoundException, ForbiddenException {
    Job job = null;
    try {
      job = this.jobRepository.first(new JobByIdSpecification(jobId));
    }
    catch(javax.persistence.NoResultException e) {
      System.err.println("Job " + jobId + " not found!");
      e.printStackTrace();
    }

    if(job == null) {
      throw new NotFoundException("Job not found: " + jobId);
    }
    if(!userProjects.contains(job.getProjectId())) {
      throw new ForbiddenException("User has no permission to edit this job");
    }
    job.setDescription(newComment);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    Job job = this.jobRepository.first(new JobByIdSpecification(jobId));
    job.setDeleted(true);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    JobsInListSpecification spec = new JobsInListSpecification(jobIds);
    List<Job> jobs = this.jobRepository.find(spec);
    for (Job job : jobs) {
      job.setDeleted(true);
      this.jobRepository.update(job);
    }
  }

  private int adjustOffset(int offset, int limit, int total) {
    // ajusta para última página, caso esteja além dela
    if (offset > 0) {
      if (offset >= total) {
        if (total % limit == 0) {
          offset = total - limit;
        }
        else {
          offset = total - (total % limit);
        }
      }
    }
    return offset;
  }

  private Sort[] toSortArray(String sortAttribute, boolean ascending) {
    if (sortAttribute == null) {
      return new Sort[0];
    }
    return new Sort[] { new Sort(sortAttribute, ascending) };
  }

  @Transactional
  public Tuple<List<Job>, Integer> findJobs(String rsqlQuery,
                                            int offset,
                                            int limit,
                                            boolean ascending,
                                            String sortAttribute) {
    JPASpecification<Job> query = jobRsqlBuilder.create(rsqlQuery);
    int total = Math.toIntExact(jobRepository.count(query));
    if (total == 0) {
      return new Tuple<>(Collections.emptyList(), 0);
    }
    offset = adjustOffset(offset, limit, total);
    Sort[] sorting = toSortArray(sortAttribute, ascending);
    List<Long> jobIds = jobRepository.findJobIds(query, limit, offset, sorting);
    List<Job> jobList = jobRepository.getJobs(jobIds, sorting);
    return new Tuple<>(jobList, total);
  }
}
