package br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository;

import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.impl.JPARepository;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;

import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Order;
import javax.persistence.criteria.Path;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

@Transactional
@Repository
public class JobRepository extends JPARepository<Job> {

  @PersistenceContext private EntityManager _entityManager;

  @Override
  public Class<Job> getType() {
    return Job.class;
  }

  @Override
  public EntityManager getEntityManager() {
    return _entityManager;
  }

  public void setEntityManager(EntityManager entityManager) {
    this._entityManager = entityManager;
  }

  public List<Job> getJobs(List<Long> ids, Sort... sorts) {
    EntityManager manager = getEntityManager();
    CriteriaBuilder builder = manager.getCriteriaBuilder();
    CriteriaQuery<Job> criteria = builder.createQuery(Job.class);
    Root<Job> root = criteria.from(Job.class);
    criteria.where(root.get("id").in(ids))
            .orderBy(createSorting(builder, root, sorts));
    return manager.createQuery(criteria).getResultList();
  }

  public List<Long> findJobIds(JPASpecification<Job> specification,
                               int maxResult,
                               int offset,
                               Sort... sorts) {
    EntityManager manager = getEntityManager();
    CriteriaBuilder builder = manager.getCriteriaBuilder();
    CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
    Root<Job> root = criteria.from(Job.class);
    Predicate predicate = specification.toPredicate(root, builder);
    criteria.select(root.get("id"))
            .where(predicate)
            .orderBy(createSorting(builder, root, sorts));
    TypedQuery<Long> query = manager.createQuery(criteria);
    query.setMaxResults(maxResult);
    if (offset > 0) query.setFirstResult(offset);
    return query.getResultList();
  }

  private <T> List<Order> createSorting(CriteriaBuilder builder,
                                          Root<T> root,
                                          Sort... sorts) {
    List<Order> sorting = new ArrayList<>();
    for (Sort sort : sorts) {
      if (sort.isAscending())
        sorting.add(builder.asc(root.get(sort.getAttribute())));
      else sorting.add(builder.desc(root.get(sort.getAttribute())));
    }
    return sorting;
  }

}
