package br.pucrio.tecgraf.soma.job.application.service;

import com.github.tennaito.rsql.misc.SimpleMapper;

import br.pucrio.tecgraf.soma.job.api.model.JobPagination;
import br.pucrio.tecgraf.soma.job.api.model.JobResponse;
import br.pucrio.tecgraf.soma.job.application.JsonUndefined;
import br.pucrio.tecgraf.soma.job.application.Tuple;
import br.pucrio.tecgraf.soma.job.domain.dto.DomainMapper;
import br.pucrio.tecgraf.soma.job.domain.model.Job;
import br.pucrio.tecgraf.soma.job.domain.model.JobAlgorithm;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobAlgorithmRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobGroupValCount;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.repository.JobRepository;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobByIdSpecification;
import br.pucrio.tecgraf.soma.job.infrastructure.persistence.specification.JobsInListSpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.repository.Sort;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;
import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl.RSQLSpecFactory;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.NotFoundException;

@Service
public class JobService {

  @Autowired private JobRepository jobRepository;
  @Autowired private JobAlgorithmRepository jobAlgorithmRepository;

  private final RSQLSpecFactory<Job> jobRsqlBuilder;
  private final RSQLSpecFactory<JobAlgorithm> jobAlgoRsqlBuilder;

  private static final SimpleMapper mapper = new SimpleMapper(1);

  static {
    Map<String, String> jobAttribAliases = new HashMap<>(10);
    jobAttribAliases.put("jobStatus"         , "statusHistory.status");
    jobAttribAliases.put("jobStatusTimestamp", "statusHistory.timestamp");
    jobAttribAliases.put("algorithmId"       , "algorithms.algorithmId");
    jobAttribAliases.put("algorithmVersion"  , "algorithms.algorithmVersion");
    jobAttribAliases.put("algorithmName"     , "algorithms.algorithmName");
    jobAttribAliases.put("flowNodeId"        , "algorithms.flowNodeId");
    jobAttribAliases.put("parameterId"       , "algorithms.parameters.parameterId");
    jobAttribAliases.put("paramLabel"        , "algorithms.parameters.label");
    jobAttribAliases.put("paramType"         , "algorithms.parameters.type");
    jobAttribAliases.put("paramValue"        , "algorithms.parameters.value");
    mapper.addMapping(Job.class, jobAttribAliases);
  }

  JobService(JobRepository jobRepository, JobAlgorithmRepository jobAlgorithmRepository) {
    this.jobRepository = jobRepository;
    this.jobAlgorithmRepository = jobAlgorithmRepository;

    this.jobRsqlBuilder = new RSQLSpecFactory<>(jobRepository.getEntityManager(), mapper);
    this.jobAlgoRsqlBuilder = new RSQLSpecFactory<>(jobAlgorithmRepository.getEntityManager());
  }

  @Transactional
  public List<JobAlgorithm> findDistinctAlgorithms(String rsqlQuery) {
    JPASpecification<JobAlgorithm> query = jobAlgoRsqlBuilder.create(rsqlQuery);
    return jobAlgorithmRepository.findDistinct(query);
  }

  @Transactional
  public void editJobComment(String jobId, String newComment, List<String> userProjects)
          throws NotFoundException, ForbiddenException {
    Job job = null;
    try {
      job = this.jobRepository.first(new JobByIdSpecification(jobId));
    }
    catch(javax.persistence.NoResultException e) {
      System.err.println("Job " + jobId + " not found!");
      e.printStackTrace();
    }

    if(job == null) {
      throw new NotFoundException("Job not found: " + jobId);
    }
    if(!userProjects.contains(job.getProjectId())) {
      throw new ForbiddenException("User has no permission to edit this job");
    }
    job.setDescription(newComment);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobAsDeleted(String jobId) {
    Job job = this.jobRepository.first(new JobByIdSpecification(jobId));
    job.setDeleted(true);
    this.jobRepository.update(job);
  }

  @Transactional
  public void markJobsAsDeleted(List<String> jobIds) {
    JobsInListSpecification spec = new JobsInListSpecification(jobIds);
    List<Job> jobs = this.jobRepository.find(spec);
    for (Job job : jobs) {
      job.setDeleted(true);
      this.jobRepository.update(job);
    }
  }

  private int adjustOffset(int offset, int limit, int total) {
    // ajusta para última página, caso esteja além dela
    if (offset > 0) {
      if (offset >= total) {
        if (total % limit == 0) {
          offset = total - limit;
        }
        else {
          offset = total - (total % limit);
        }
      }
    }
    return offset;
  }

  private Sort[] toSortArray(String sortAttribute, boolean ascending) {
    if (sortAttribute == null) {
      return new Sort[0];
    }
    return new Sort[] { new Sort(sortAttribute, ascending) };
  }

  @Transactional
  public void findJobs(String rsqlQuery,
                       int limit,
                       int offset,
                       boolean ascending,
                       String sortAttribute,
                       JobResponse response) {
    JobPagination pagination = response.getPagination();
    JPASpecification<Job> query = jobRsqlBuilder.create(rsqlQuery);
    int total = Math.toIntExact(jobRepository.count(query));
    pagination.setTotal(total);
    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    }
    else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);
      List<Long> jobIds = jobRepository.findJobIds(query, limit, offset, sorting);
      List<Job> jobList = jobRepository.getJobs(jobIds, sorting);
      pagination.setOffset(offset);
      response.getData().setJobs(
              Arrays.asList(DomainMapper.convert(jobList,
                      br.pucrio.tecgraf.soma.job.api.model.Job[].class)));
    }
  }

  @Transactional
  public void findGroupedJobs(String rsqlQuery,
                              int limit,
                              int offset,
                              boolean ascending,
                              String sortAttribute,
                              JobResponse response) {
    JobPagination pagination = response.getPagination();
    JPASpecification<Job> query = jobRsqlBuilder.create(rsqlQuery);
    int total = Math.toIntExact(jobRepository.countGroups(query));
    pagination.setTotal(total);
    if (total == 0) {
      pagination.setOffset(0);
      response.getData().setJobs(Collections.emptyList());
    }
    else {
      offset = adjustOffset(offset, limit, total);
      Sort[] sorting = toSortArray(sortAttribute, ascending);
      List<JobGroupValCount> groups = jobRepository.findGroupedJobs(query, limit, offset, sorting);
      List<String> groupIds = new ArrayList<>(groups.size());
      Map<Long, Integer> jobId2Pos = new HashMap<>(groups.size());
      for (int pos = 0; pos < groups.size(); pos++) {
        JobGroupValCount group = groups.get(pos);
        jobId2Pos.put(group.firstJobId, pos);
        groupIds.add(group.groupId);
      }
      groupIds = jobRepository.filterSingletons(groupIds);
      List<Job> jobList = jobRepository.getJobs(jobId2Pos.keySet());
      br.pucrio.tecgraf.soma.job.api.model.Job[] jsonJobs =
              new br.pucrio.tecgraf.soma.job.api.model.Job[groups.size()];
      for (Job job: jobList) {
        br.pucrio.tecgraf.soma.job.api.model.Job jsonJob =
                DomainMapper.convert(job, br.pucrio.tecgraf.soma.job.api.model.Job.class);
        int i = jobId2Pos.get(job.getId());
        jsonJobs[i] = jsonJob;

        if (groupIds.contains(job.getGroupId())) {
          JobGroupValCount group = groups.get(i);
          jsonJob.setIsGroup(true);
          jsonJob.setJobId(JsonUndefined.STRING);
          jsonJob.setEndTime(JsonUndefined.STRING);
          jsonJob.setCpuTime(JsonUndefined.DOUBLE);
          jsonJob.setWallclockTime(JsonUndefined.INTEGER);
          jsonJob.setRamMemory(JsonUndefined.DOUBLE);
          jsonJob.setAlgorithms(JsonUndefined.typedList());
          jsonJob.setStatusHistory(JsonUndefined.typedList());

          jsonJob.setSubmissionTime(group.submissionTime.toString());
          jsonJob.setLastModifiedTime(group.lastModifiedTime.toString());

          // clear heterogeneous values.
          if (group.projectId > 1) jsonJob.setProjectId(JsonUndefined.STRING);
          if (group.jobOwner > 1) jsonJob.setJobOwner(JsonUndefined.STRING);
          if (group.automaticallyMachineSelection > 1)
            jsonJob.setAutomaticallyMachineSelection(JsonUndefined.BOOLEAN);
          if (group.numberOfProcesses > 1) jsonJob.setNumberOfProcesses(JsonUndefined.INTEGER);
          if (group.numberOfProcessesByMachine > 1)
            jsonJob.setNumberOfProcessesByMachine(JsonUndefined.INTEGER);
          if (group.description > 1) jsonJob.setDescription(JsonUndefined.STRING);
          if (group.priority > 1) jsonJob.setPriority(JsonUndefined.INTEGER);
          if (group.multipleExecution > 1) jsonJob.setMultipleExecution(JsonUndefined.BOOLEAN);
          if (group.jobType > 1) jsonJob.setJobType(null);
          if (group.executionMachine > 1) jsonJob.setExecutionMachine(JsonUndefined.STRING);
          if (group.exitCode > 1) jsonJob.setExitCode(JsonUndefined.INTEGER);
          if (group.guiltyNodeId > 1) jsonJob.setGuiltyNodeId(JsonUndefined.STRING);
          if (group.exitStatus > 1) jsonJob.setExitStatus(null);
          if (group.flowId > 1) jsonJob.setFlowId(JsonUndefined.STRING);
          if (group.flowVersion > 1) jsonJob.setFlowVersion(JsonUndefined.STRING);
          if (group.flowName > 1) jsonJob.setFlowName(JsonUndefined.STRING);
        }
      }
      pagination.setOffset(offset);
      response.getData().setJobs(Arrays.asList(jsonJobs));
    }
  }
}
