package br.pucrio.tecgraf.soma.job.application.controller;

import java.util.List;

import javax.persistence.NoResultException;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.ServiceUnavailableException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;

import org.jboss.logging.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.transaction.CannotCreateTransactionException;

import br.pucrio.tecgraf.soma.job.api.MultiflowApi;
import br.pucrio.tecgraf.soma.job.api.model.Multiflow;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowBasicResponse;
import br.pucrio.tecgraf.soma.job.api.model.MultiflowFullResponse;
import br.pucrio.tecgraf.soma.job.api.model.ReplicaFile;
import br.pucrio.tecgraf.soma.job.application.appservice.MultiflowAppService;

@Component
public class MultiflowController extends MultiflowApi {
  private static final Logger LOG = Logger.getLogger(MultiflowController.class);

  @Autowired private HttpServletRequest request;
  @Autowired private MultiflowAppService multiflowAppService;

  public MultiflowController() {
    super(null);
  }

  @Override
  public Response multiflowPost(Multiflow multiflow, String locale, SecurityContext securityContext) {
    MultiflowBasicResponse multiflowResponse;

    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    try {
      multiflowResponse = multiflowAppService.createMultiflow(multiflow);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (Exception e) { // Demais erros
      String errorMsg = "Error creating a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.CREATED.value()).entity(multiflowResponse).build();
  }

  @Override
  public Response multiflowGet(String projectId, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    List<MultiflowFullResponse> multiflows;

    try {
      multiflows = multiflowAppService.findMultiflowsByProjectId(projectId);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflows).build();
  }

  @Override
  public Response multiflowMultiflowIdGet(Long multiflowId, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    MultiflowFullResponse multiflow;

    try {
      multiflow = multiflowAppService.findMultiflowById(multiflowId);
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (NoResultException e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error getting a Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflow).build();
  }

  @Override
  public Response multiflowMultiflowIdPatch(Long multiflowId, ReplicaFile replicaFile, String locale, SecurityContext securityContext) {
    String accessToken = request.getHeader(HttpHeaders.AUTHORIZATION);
    if (accessToken == null || accessToken.isEmpty()) {
      LOG.info("Empty or null token");
      return Response.status(HttpStatus.UNAUTHORIZED.value()).build();
    }

    MultiflowFullResponse multiflow;

    try {
      multiflow = multiflowAppService.updateMultiflow(multiflowId, replicaFile.getReplicaFilePath(), replicaFile.getReplicaFileName());
    } catch (ServiceUnavailableException | CannotCreateTransactionException e) { // Casos de falha de conexão com o BD
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.SERVICE_UNAVAILABLE.value()).entity(errorMsg).build();
    } catch (ForbiddenException e) { // Casos de falta de permissão no projeto
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.FORBIDDEN.value()).entity(errorMsg).build();
    } catch (NoResultException e) {
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.NOT_FOUND.value()).entity(errorMsg).build();
    } catch (Exception e) {
      String errorMsg = "Error updating Multiflow: " + e.getMessage();
      LOG.info(errorMsg);
      return Response.status(HttpStatus.INTERNAL_SERVER_ERROR.value()).entity(errorMsg).build();
    }

    return Response.status(HttpStatus.OK.value()).entity(multiflow).build();
  }
}
