package br.pucrio.tecgraf.soma.websocketnotifier.application.service;

import br.pucrio.tecgraf.soma.serviceapi.configuration.ServiceConfiguration;
import br.pucrio.tecgraf.soma.websocketnotifier.application.configuration.Constants;
import br.pucrio.tecgraf.soma.websocketnotifier.infrastructure.persistence.message.KafkaEventReader;
import br.pucrio.tecgraf.soma.websocketnotifier.model.WebsocketNotifierInfo;
import br.pucrio.tecgraf.soma.websocketnotifier.model.WebsocketNotifierStatus;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.StringSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static org.apache.kafka.clients.producer.ProducerConfig.*;

@Service
public class ProducerHandler {
    private final Logger logger = LoggerFactory.getLogger(ProducerHandler.class);

    private Producer<String, String> producer;

    private final String KAFKA_SERVER;
    private final String KAFKA_NOTIFICATION_TOPIC;
    private final List<String> KAFKA_TOPICS;
    private final String KAFKA_SCHEMA_REGISTRY_URL;
    private final List<Integer> LAST_EVENTS_MINIMUM_NUMBER;

    private KtableService ktableService;

    private KafkaEventReader eventReader;

    @Autowired
    public ProducerHandler(ServiceConfiguration serviceConfiguration, KtableService ktableService,KafkaEventReader eventReader) {
        this.KAFKA_SERVER = serviceConfiguration.getValue(Constants.Config.KAFKA_SERVER_ADDRESS.option.getLongName());
        this.KAFKA_NOTIFICATION_TOPIC = serviceConfiguration.getValue(Constants.Config.KAFKA_NOTIFICATION_TOPIC.option.getLongName());
        this.KAFKA_TOPICS = Arrays.asList(serviceConfiguration.getValue(Constants.Config.KAFKA_TOPIC.option.getLongName()).split(";"));
        this.KAFKA_SCHEMA_REGISTRY_URL = serviceConfiguration.getValue(Constants.Config.KAFKA_SCHEMA_REGISTRY_URL.option.getLongName());
        this.LAST_EVENTS_MINIMUM_NUMBER = Arrays.stream(serviceConfiguration.getValue(Constants.Config.LAST_EVENTS_MINIMUM_NUMBER.option.getLongName()).split(";")).map(Integer::parseInt).collect(Collectors.toList());
        this.ktableService = ktableService;
        this.eventReader = eventReader;

        if (KAFKA_TOPICS.size() != LAST_EVENTS_MINIMUM_NUMBER.size()) {
          String msg =
                  
                          "List arguments value of --%s and --%s must have the same length".formatted(
                          Constants.Config.KAFKA_TOPIC.option.getLongName(),
                          Constants.Config.LAST_EVENTS_MINIMUM_NUMBER.option.getLongName());
          logger.error(msg);
          throw new IllegalArgumentException(msg);
        }
    }

    public WebsocketNotifierInfo getLastOffset(WebsocketNotifierInfo wsInfo){
        return this.ktableService.getLastUserOffset(wsInfo);
    }

    protected Producer<String, String> getProducer() {
        if(producer == null) {
            Properties properties = new Properties();
            properties.setProperty(BOOTSTRAP_SERVERS_CONFIG, KAFKA_SERVER);
            properties.setProperty(KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
            properties.setProperty(VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class.getName());
            properties.setProperty(ACKS_CONFIG, "1");
            properties.setProperty(RETRIES_CONFIG, "3");
            properties.setProperty(LINGER_MS_CONFIG, "1");
            producer = new KafkaProducer<>(properties);
        }
        return producer;
    }

    public void publishKafkaOffset(WebsocketNotifierInfo message, WebsocketNotifierStatus status){
        if(!status.equals(WebsocketNotifierStatus.OLD)) {
            String eventMessage = null;
            JsonObject jsonObject = null;
            try {
                Gson gson = new Gson();
                eventMessage = gson.toJson(message);
                JsonParser parser = new JsonParser();
                jsonObject = (JsonObject) parser.parse(eventMessage);
            } catch (Error err) {
                logger.error("Error during push event: " + message.toString(), err);
            }
            if (eventMessage != null && !eventMessage.isEmpty()) {
                this.getProducer().send(new ProducerRecord<>(this.KAFKA_NOTIFICATION_TOPIC, jsonObject.get("user").getAsString(), jsonObject.toString()));
                logger.debug("Published topic {} user {}", this.KAFKA_NOTIFICATION_TOPIC, jsonObject);
            }
        }
    }

    public List<WebsocketNotifierInfo> buildWebsocketInfos(String user, String application, List<String> topics) {
        List<WebsocketNotifierInfo> wsInfoList = new ArrayList<>();
        for (String topic_name: topics) {
            WebsocketNotifierInfo wsInfo = new WebsocketNotifierInfo();
            wsInfo.setTopicName(topic_name);
            wsInfo.setUser(user);
            wsInfo.setApplication(application);
            wsInfo = this.getLastOffset(wsInfo);
            wsInfoList.add(wsInfo);
        }
        return wsInfoList;
    }



    private int getLastMinimumNumberParam(CharSequence topic) {
        int minimumOffset = 0;
        for (int i = 0; i < this.KAFKA_TOPICS.size(); i++) {
            if(this.KAFKA_TOPICS.get(i).equals(topic)){
                minimumOffset = this.LAST_EVENTS_MINIMUM_NUMBER.get(i);
                break;
            }
        }
        if(minimumOffset < 0){
            minimumOffset = 0;
        }
        return minimumOffset;
    }

    private int getLastMinimumNumber(String topic, int currentUserOffset){
        // 10
        int minimumOffset = this.getLastMinimumNumberParam(topic);
        currentUserOffset = currentUserOffset - minimumOffset;
        if(currentUserOffset > 0){
            // decremento -1 porque o kafka retorna o evento da posição (inclusive), logo seria minimumOffset + posição atual que é maior que minimumOffset
            return currentUserOffset + 1;
        } else {
            return 0;
        }
    }

    private void publishUserFirstOffset(Consumer<String, Object> consumer, WebsocketNotifierInfo wsInfo){
        Set<TopicPartition> assignment = consumer.assignment();
        long maxOffset = 1;
        for (TopicPartition tp : assignment) {
            if (consumer.position(tp) > maxOffset) {
                maxOffset = consumer.position(tp);
            }
        }
        wsInfo.setOffset((int) maxOffset);
        this.publishKafkaOffset(wsInfo, WebsocketNotifierStatus.LIVE);
    }

    public List<JsonObject> getLostMessagesToUser(List<WebsocketNotifierInfo> wsInfos) {
        List<JsonObject> recordFromOffset = new ArrayList<>();
        if(wsInfos!= null) {
            for (WebsocketNotifierInfo wsInfo : wsInfos) {
                Consumer<String, Object> consumer = eventReader.buildKafkaConsumer(KAFKA_SERVER, KAFKA_SCHEMA_REGISTRY_URL, (String) wsInfo.getTopicName(), wsInfo.getTopicName() + "-lostevents-" + System.currentTimeMillis());
                int beginOffset = getLastMinimumNumber(wsInfo.getTopicName().toString(), wsInfo.getOffset());
                List<JsonObject> events = eventReader.readRecordsFromOffset(consumer, wsInfo, beginOffset);
                if(!events.isEmpty()){
                    recordFromOffset.addAll(events);
                } else {
                    this.publishUserFirstOffset(consumer, wsInfo);
                }
                consumer.close();
            }
        }
        return recordFromOffset;
    }

}
