/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.IntIterable;
import com.gs.collections.api.LazyIntIterable;
import com.gs.collections.api.block.function.primitive.IntToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.IntPredicate;
import com.gs.collections.api.iterator.IntIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableIntSet;
import com.gs.collections.api.set.primitive.IntSet;
import com.gs.collections.api.set.primitive.MutableIntSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedIntCollection;
import com.gs.collections.impl.lazy.primitive.LazyIntIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableIntSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link IntIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 *
 * @see MutableIntSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedIntSet
        extends AbstractSynchronizedIntCollection
        implements MutableIntSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedIntSet(MutableIntSet set)
    {
        super(set);
    }

    SynchronizedIntSet(MutableIntSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableIntSet and wrap it directly in a SynchronizedIntSet.
     */
    public static SynchronizedIntSet of(MutableIntSet set)
    {
        return new SynchronizedIntSet(set);
    }

    /**
     * This method will take a MutableIntSet and wrap it directly in a SynchronizedIntSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedIntSet of(MutableIntSet set, Object lock)
    {
        return new SynchronizedIntSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableIntSet getMutableIntSet()
    {
        return (MutableIntSet) this.getIntCollection();
    }

    @Override
    public SynchronizedIntSet without(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntSet with(int element)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedIntSet withAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedIntSet withoutAll(IntIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableIntSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableIntSet select(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().select(predicate);
        }
    }

    @Override
    public MutableIntSet reject(IntPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(IntToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().hashCode();
        }
    }

    @Override
    public LazyIntIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyIntIterableAdapter(this);
        }
    }

    @Override
    public MutableIntSet asUnmodifiable()
    {
        return new UnmodifiableIntSet(this);
    }

    @Override
    public MutableIntSet asSynchronized()
    {
        return this;
    }

    public IntSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().freeze();
        }
    }

    @Override
    public ImmutableIntSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableIntSet().toImmutable();
        }
    }
}
