/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.bag.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.bag.MutableBag;
import com.gs.collections.api.bag.primitive.ImmutableFloatBag;
import com.gs.collections.api.bag.primitive.MutableFloatBag;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.api.block.procedure.primitive.FloatIntProcedure;
import com.gs.collections.api.iterator.FloatIterator;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedFloatCollection;
import com.gs.collections.impl.factory.primitive.FloatBags;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableFloatBag}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link FloatIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveBag.stg.
 *
 * @see MutableFloatBag#asSynchronized()
 * @see MutableBag#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedFloatBag
        extends AbstractSynchronizedFloatCollection
        implements MutableFloatBag
{
    private static final long serialVersionUID = 1L;

    SynchronizedFloatBag(MutableFloatBag bag)
    {
        super(bag);
    }

    SynchronizedFloatBag(MutableFloatBag bag, Object newLock)
    {
        super(bag, newLock);
    }

    @GuardedBy("getLock()")
    private MutableFloatBag getMutableFloatBag()
    {
        return (MutableFloatBag) this.getFloatCollection();
    }

    @Override
    public SynchronizedFloatBag with(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatBag without(float element)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedFloatBag withAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedFloatBag withoutAll(FloatIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().removeAll(elements);
        }
        return this;
    }

    public void addOccurrences(float item, int occurrences)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().addOccurrences(item, occurrences);
        }
    }

    public boolean removeOccurrences(float item, int occurrences)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().removeOccurrences(item, occurrences);
        }
    }

    public int sizeDistinct()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().sizeDistinct();
        }
    }

    public int occurrencesOf(float item)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().occurrencesOf(item);
        }
    }

    public void forEachWithOccurrences(FloatIntProcedure procedure)
    {
        synchronized (this.getLock())
        {
            this.getMutableFloatBag().forEachWithOccurrences(procedure);
        }
    }

    @Override
    public MutableFloatBag select(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().select(predicate);
        }
    }

    @Override
    public MutableFloatBag reject(FloatPredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().reject(predicate);
        }
    }

    @Override
    public <V> MutableBag<V> collect(FloatToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherBag)
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().equals(otherBag);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableFloatBag().hashCode();
        }
    }

    @Override
    public LazyFloatIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyFloatIterableAdapter(this);
        }
    }

    @Override
    public MutableFloatBag asUnmodifiable()
    {
        return new UnmodifiableFloatBag(this);
    }

    @Override
    public MutableFloatBag asSynchronized()
    {
        return this;
    }

    @Override
    public ImmutableFloatBag toImmutable()
    {
        return FloatBags.immutable.withAll(this);
    }
}
